<?php
/**
 * AJAX endpoint pour sauvegarder les modifications ODS
 */

include_once('../../../inc/includes.php');

// Vérifier les permissions et la session
Session::checkLoginUser();

header('Content-Type: application/json');

try {
    // Vérifier que c'est bien une requête POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Méthode non autorisée');
    }

    // Vérifier les paramètres requis
    if (!isset($_POST['document_id']) || !isset($_POST['cells'])) {
        throw new Exception('Paramètres manquants');
    }

    $document_id = (int) $_POST['document_id'];
    
    // Décoder les cellules depuis JSON
    $cells_json = $_POST['cells'];
    $cells = json_decode($cells_json, true);
    
    if ($cells === null) {
        throw new Exception('Format de données invalide');
    }

    // Charger le document
    $document = new Document();
    if (!$document->getFromDB($document_id)) {
        throw new Exception('Document non trouvé');
    }

    // Vérifier les droits d'écriture sur le document
    if (!$document->canUpdateItem()) {
        throw new Exception('Permissions insuffisantes');
    }

    // Construire le chemin du fichier ODS
    $file = GLPI_DOC_DIR . '/' . $document->fields['filepath'];
    if (!is_file($file) || !is_readable($file)) {
        throw new Exception('Fichier ODS introuvable ou inaccessible');
    }

    // Détection du type de fichier et mapping vers ODS
    $ext = strtolower(pathinfo($file, PATHINFO_EXTENSION));
    $isExcel = in_array($ext, ['xls','xlsx','xlsm'], true);
    $isOds   = ($ext === 'ods');

    if ($isExcel) {
        $odsFile = preg_replace('~\.(xlsx|xlsm|xls)$~i', '.ods', $file);
    } elseif ($isOds) {
        $odsFile = $file;
    } else {
        throw new Exception('Type de fichier non supporté pour l\'édition');
    }

    if (!is_file($odsFile) || !is_readable($odsFile)) {
        throw new Exception('Fichier ODS parallèle introuvable');
    }

    // Utiliser la méthode saveOdsEdits du Document
    $newFile = $document->saveOdsEdits($odsFile, $cells);

    // Réponse de succès
    echo json_encode([
        'success' => true,
        'message' => 'Modifications enregistrées avec succès',
        'new_file' => basename($newFile),
        'timestamp' => time()
    ]);

} catch (Exception $e) {
    // Réponse d'erreur
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage(),
        'error' => true
    ]);
} catch (Throwable $e) {
    // Erreur inattendue
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erreur serveur : ' . $e->getMessage(),
        'error' => true
    ]);
} 