<?php
use Glpi\Event;

class PluginDlteamsObjectCollection extends CommonDBTM {
    public const MOVE_BEFORE = 'before';
    public const MOVE_AFTER  = 'after';

    /** Nom de la colonne d’ordre (à surcharger en enfant) */
    protected $orderColumn = 'ranking';

    /**
     * Clé étrangère vers l’objet parent (à surcharger en enfant, ex. 'records_id').
     */
    static public $items_id_1;

    /**
     * Réindexe (1..N sans trou) les enregistrements filtrés par $extraCriteria.
     */
    public function normalizeOrder(CommonGLPI $itemModel, array $extraCriteria = []): bool {
        global $DB;
        $ordered = $this->getOrderedList($itemModel, $extraCriteria);
        $pos     = 1;
        $DB->beginTransaction();
        try {
            foreach ($ordered as $row) {
                if ((int)$row[$this->orderColumn] !== $pos) {
                    $DB->update(
                        $itemModel->getTable(),
                        [$this->orderColumn => $pos],
                        ['id'                => (int)$row['id']]
                    );
                }
                $pos++;
            }
            $DB->commit();
            return true;
        } catch (\Exception $e) {
            $DB->rollback();
            return false;
        }
    }

    /**
     * Retourne la liste ordonnée de [ 'id'=>…, '<orderColumn>'=>… ].
     */
    public function getOrderedList(CommonGLPI $itemModel, array $extraCriteria = []): array {
        global $DB;
        $query = [
            'SELECT' => ['id', $this->orderColumn],
            'FROM'   => $itemModel->getTable(),
            'ORDER'  => $this->orderColumn
        ];
        if (!empty($extraCriteria)) {
            $query['WHERE'] = $extraCriteria;
        }
        $out = [];
        foreach ($DB->request($query) as $row) {
            $out[] = $row;
        }
        return $out;
    }

    /**
     * Réordonnancement générique d’un item (0-based → 1-based),
     * via un appel dynamique à une méthode (ex. getRequest).
     *
     * @param int    $oldZero             ancien index 0-based
     * @param int    $newZero             nouvel index 0-based
     * @param string $direction           self::MOVE_BEFORE|MOVE_AFTER
     * @param string $sourceClass         nom de la classe à appeler
     * @param string $sourceMethod        nom de la méthode statique
     * @param array  $sourceArgs          arguments à passer à la méthode
     * @param bool   $methodReturnsQuery  true si la méthode renvoie un tableau de critères GLPI
     */
    public function reorderByIndex(
        int    $oldZero,
        int    $newZero,
        string $direction,
        string $sourceClass,
        string $sourceMethod,
        array  $sourceArgs = [],
        bool   $methodReturnsQuery = false
    ): bool {
        global $DB;

        // 1) conversion 0-based → 1-based
        $oldPos = $oldZero + 1;
        $newPos = $newZero + 1;

        // 2) réindexation préalable pour combler les trous
        $filter = [ static::$items_id_1 => $this->fields[static::$items_id_1] ];
        $this->normalizeOrder($this, $filter);

        // 3) récupération de la liste via appel dynamique
        if ($methodReturnsQuery) {
            $criteria = call_user_func_array([$sourceClass, $sourceMethod], $sourceArgs);
            $iterator = $DB->request($criteria);
            $items    = [];
            foreach ($iterator as $r) {
                $items[] = [
                    'id'       => $r['linkid'],
                    'position' => (int)$r[$this->orderColumn]
                ];
            }
        } else {
            $items = call_user_func_array([$sourceClass, $sourceMethod], $sourceArgs);
        }

        // 4) extraction des IDs et positions d’origine
        $ids      = array_column($items, 'id');
        $origPos  = array_column($items, 'position', 'id');

        // 5) retrait et insertion à la nouvelle place
        $movedId = (int)$this->fields['id'];
        $offset  = array_search($movedId, $ids, true);
        if ($offset === false) {
            return false;
        }
        array_splice($ids, $offset, 1);
        array_splice($ids, $newPos - 1, 0, $movedId);

        // 6) mise à jour dans une transaction
        $DB->beginTransaction();
        try {
            foreach ($ids as $i => $id) {
                $pos = $i + 1;
                if ($origPos[$id] !== $pos) {
                    $DB->update(
                        $this->getTable(),
                        [$this->orderColumn => $pos],
                        ['id'                => $id]
                    );
                }
            }
            $DB->commit();
            return true;
        } catch (\Exception $e) {
            $DB->rollback();
            return false;
        }
    }

    /**
     * Hook GLPI après suppression, pour réindexer automatiquement.
     * À enregistrer dans votre init() :
     *   Event::register(\"item.delete\", [PluginDlteamsObjectCollection::class, 'onItemDelete']);
     */
    public static function onItemDelete(CommonDBTM $item) {
        $col = new self();
        $col->normalizeOrder($item, [static::$items_id_1 => $item->fields[static::$items_id_1]]);
    }
}
