<?php

/**
 * ---------------------------------------------------------------------
 *
 * GLPI - Gestionnaire Libre de Parc Informatique
 *
 * http://glpi-project.org
 *
 * @copyright 2015-2025 Teclib' and contributors.
 * @copyright 2003-2014 by the INDEPNET Development Team.
 * @licence   https://www.gnu.org/licenses/gpl-3.0.html
 *
 * ---------------------------------------------------------------------
 *
 * LICENSE
 *
 * This file is part of GLPI.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * ---------------------------------------------------------------------
 */

use Glpi\Event;
use Glpi\Toolbox\Sanitizer;

/**
 * Document class
 **/
class Document extends CommonDBVisible implements ExtraVisibilityCriteria
{
    use Glpi\Features\TreeBrowse;
    use Glpi\Features\ParentStatus;

//    dlteams
    use \Glpi\Features\Clonable;

//    dlteams
    const READFAQ       = 2048;
    const PUBLISHFAQ    = 4096;
    const COMMENTS      = 8192;

    // From CommonDBTM
    public $dohistory                   = true;

    protected static $forward_entity_to = ['Document_Item'];

    public static $rightname                   = 'document';
    public static $tag_prefix                  = '#';
    protected $usenotepad               = true;


    public static function getTypeName($nb = 0)
    {
        return _n('Document', 'Documents', $nb);
    }

    public function getCloneRelations(): array
    {
        return [
            Entity_KnowbaseItem::class,
            Group_KnowbaseItem::class,
            KnowbaseItem_Profile::class,
            KnowbaseItem_User::class,
        ];
    }

    /**
     * Check if given object can have Document
     *
     * @since 0.85
     *
     * @param string|object $item An object or a string
     *
     * @return boolean
     **/
    public static function canApplyOn($item)
    {
        /** @var array $CFG_GLPI */
        global $CFG_GLPI;

        // All devices can have documents!
        if (
            is_a($item, 'Item_Devices', true)
            || is_a($item, 'CommonDevice', true)
        ) {
            return true;
        }

        // We also allow direct items to check
        if ($item instanceof CommonGLPI) {
            $item = $item->getType();
        }

        if (in_array($item, $CFG_GLPI['document_types'])) {
            return true;
        }

        return false;
    }

    /**
     * Return visibility joins to add to DBIterator parameters
     *
     * @since 9.2
     *
     * @param boolean $forceall force all joins (false by default)
     *
     * @return array
     */
    public static function getVisibilityCriteria(bool $forceall = false): array
    {
        /** @var array $CFG_GLPI */
        global $CFG_GLPI;

        // Build common JOIN clause
        $criteria = [
            'LEFT JOIN' => self::getVisibilityCriteriaCommonJoin($forceall),
        ];

        // Handle anonymous users
        if (!Session::getLoginUserID()) {
            if ($CFG_GLPI["use_public_faq"]) {
                // Public FAQ is enabled; show FAQ
                $criteria['WHERE'] = self::getVisibilityCriteriaFAQ();
                return $criteria;
            } else {
                // Public FAQ is disabled; show nothing
                $criteria['WHERE'] = [0];
                return $criteria;
            }
        }

        // Handle logged in users
        if (Session::getCurrentInterface() == "helpdesk") {
            // Show FAQ for helpdesk user
            $criteria['WHERE'] = self::getVisibilityCriteriaFAQ();
            return $criteria;
        } else {
            // Show knowledge base for central users
            $criteria['WHERE'] = self::getVisibilityCriteriaKB();
            return $criteria;
        }
    }

    public static function showCentralList($start = 0, $status = 'process', $showgrouptickets = true)
    {
        global $DB;

        if (!Session::haveRight('document', READ)) {
            return false;
        }

        $iterator = $DB->request([
            'FROM' => self::getTable(),
            'WHERE' => [
                'entities_id' => Session::getActiveEntity()
            ],
            'ORDER' => ['date_mod DESC'],
            'LIMIT' => $_SESSION['glpidisplay_count_on_home'] ?? 10
        ]);

        $total_row_count = count($iterator);
        $displayed_row_count = min((int)$_SESSION['glpidisplay_count_on_home'], $total_row_count);

        if ($total_row_count > 0) {
            // Titre principal avec lien
            $main_header = "<a href=\"" . self::getSearchURL() . "\">" .
                Html::makeTitle(__('Recent Documents', 'dlteams'), $displayed_row_count, $total_row_count) . "</a>";

            echo "<div class='table-responsive'>";
            echo "<table class='table table-borderless table-striped table-hover card-table'>";
            echo "<thead>";
            echo "<tr>";
            echo "<th colspan='3'>" . $main_header . "</th>";
            echo "</tr>";
            echo "<tr>";
            echo "<th>" . __('ID') . "</th>";
            echo "<th>" . __('Name') . "</th>";
            echo "<th>" . __('Date modification') . "</th>";
            echo "</tr>";
            echo "</thead>";
            echo "<tbody>";

            foreach ($iterator as $data) {
                $url = self::getFormURLWithID($data['id']);
                echo "<tr>";
                echo "<td><a href='" . $url . "' class='btn btn-sm btn-outline-primary'>" . $data['id'] . "</a></td>";
                echo "<td>" . htmlspecialchars($data['name']??"")??"--" . "</td>";
                echo "<td class='text-muted'>" . Html::convDateTime($data['date_mod']) . "</td>";
                echo "</tr>";
            }

            echo "</tbody>";
            echo "</table>";
            echo "</div>";

            if ($total_row_count > $displayed_row_count) {
                echo "<div class='text-center mt-2'>";
                echo "<a href='" . self::getSearchURL() . "' class='btn btn-sm btn-outline-secondary'>";
                echo sprintf(__('Show all %d items'), $total_row_count);
                echo "</a>";
                echo "</div>";
            }
        } else {
            echo "<div class='text-center text-muted p-4'>";
            echo "<i class='fas fa-inbox fa-2x mb-2'></i><br>";
            echo __('No item found');
            echo "</div>";
        }
    }

    /**
     * Get visibility criteria for articles displayed in the FAQ (seen by
     * helpdesk and anonymous users)
     * This mean any KB article tagged as 'is_faq' should be displayed
     *
     * @return array WHERE clause
     */
    private static function getVisibilityCriteriaFAQ(): array
    {
        // Specific case for anonymous users + multi entities
        if (!Session::getLoginUserID()) {
            $where = ['is_faq' => 1];
            if (Session::isMultiEntitiesMode()) {
                $where[Entity_KnowbaseItem::getTableField('entities_id')] = 0;
                $where[Entity_KnowbaseItem::getTableField('is_recursive')] = 1;
            }
        } else {
            $where = self::getVisibilityCriteriaKB();
            $where['is_faq'] = 1;
        }

        return $where;
    }


    /**
     * Get all the types that can have a document
     *
     * @since 0.85
     *
     * @return array of the itemtypes
     **/
    public static function getItemtypesThatCanHave()
    {
        /** @var array $CFG_GLPI */
        global $CFG_GLPI;

        return array_merge(
            $CFG_GLPI['document_types'],
            CommonDevice::getDeviceTypes(),
            Item_Devices::getDeviceTypes()
        );
    }


    /**
     * @see CommonGLPI::getMenuShorcut()
     *
     * @since 0.85
     **/
    public static function getMenuShorcut()
    {
        return 'd';
    }

//        fork dlteams

    public static function getAdditionalMenuLinks()
    {
        /** @var array $CFG_GLPI */
        global $CFG_GLPI;
        if(!class_exists(PluginDlteamsObject_document::class))
            return;

//        $links = parent::getAdditionalMenuLinks();

        $icon = Config::getConfigurationValue("dlteams", "document_manage_folders")?"fa-toggle-on":"fa-toggle-off";
        $pic_tasks = "
            <span>&nbsp;
               " . __s('Dossiers') . "
            </span>
         ";

        if(class_exists(PluginDlteamsObject_document::class))
            $links[$pic_tasks] = DocumentCategory::getSearchURL(false);



        echo "<script>
                        \$(document).ready(function () {
                            // Attendre que la page soit complètement chargée
                            \$('a[href^=\"/front/documentcategory.php\"]').attr('target', '_blank');
                        });
                    </script>";



        $documents_id = static::getCurrentId();
        if($documents_id) {
            $document = new Document();
            $document->getFromDB($documents_id);

            if (strpos($_SERVER['REQUEST_URI'], 'document') !== false && isset($document->fields["is_html"]) && $document->fields["is_html"]) {
                $plus = "<i class='fas fa-add' onclick='' title='" . __("add tab", 'dlteams') . "'></i> <span style='margin-left: 4px;'>ajouter chapitre</span>";

                $temp_url = "/marketplace/dlteams/front/policieform_section.form.php" . "?items_id=" . static::getCurrentId() . "&itemtype=" . static::class . "&add_tab=" . true;

                $links[$plus] = $temp_url;
            }
        }


        return $links;
    }


    public static function getCurrentId()
    {
        // Get the current URL
        $currentURL = $_SERVER['REQUEST_URI'];

        // Get the query string from the URL
        $queryString = parse_url($currentURL, PHP_URL_QUERY);

        // Initialize an empty array for query parameters
        $queryParams = [];

        // Parse the query string into an array of parameters if it exists
        if ($queryString) {
            parse_str($queryString, $queryParams);
        }

        // Get the value of the "id" parameter
        $id = isset($queryParams['id']) ? $queryParams['id'] : null;

        return $id;
    }


//    fork dlteams
    public function getTabNameForItem(CommonGLPI $item, $withtemplate = 0)
    {

        if (!$withtemplate) {
            $nb = 0;
            switch ($item->getType()) {
                case __CLASS__:
                    /** @var Document $item */
                    $ong[1] = $this->getTypeName(1);

                    if ($item->canUpdateItem()) {
                        if ($_SESSION['glpishow_count_on_tabs']) {
                            $nb = $item->countVisibilities();
                        }
                        $ong[2] = __('Aperçu');
                        $ong[3] = self::createTabEntry(
                            __('Accès, Partages', "dlteams"),
                            $nb
                        );
                    }
                    return $ong;
            }
        }
        return '';
    }

//    fork dlteams
    public static function displayTabContentForItem(CommonGLPI $item, $tabnum = 1, $withtemplate = 0)
    {

        if ($item->getType() == __CLASS__) {
            /** @var Document $item */
            switch ($tabnum) {
                case 1:
//                    $item->showFull();
                    $item->showForm($item->getID());
                    break;

                case 2:
                    $item->showDocumentPreview($item->getID());
                    break;

                case 3:
                    $item->showVisibility();
                    break;
            }
        }
        return true;
    }

//    fork dlteams
    /**
     * Print out (html) show item : question and answer
     *
     * @param $options      array of options
     *
     * @return boolean|string
     **/
    public function showFull($options = [])
    {
        /**
         * @var array $CFG_GLPI
         * @var \DBmysql $DB
         */
        global $CFG_GLPI, $DB;

        if (!$this->can($this->fields['id'], READ)) {
            return false;
        }

        $default_options = [
            'display' => true,
        ];
        $options = array_merge($default_options, $options);

        $out = "";



        if ($this->fields["date_creation"]) {
            //TRANS: %s is the datetime of update
            $out .= sprintf(__('Created on %s'), Html::convDateTime($this->fields["date_creation"]));
            $out .= "<br>";
        }
        if ($this->fields["date_mod"]) {
            //TRANS: %s is the datetime of update
            $out .= sprintf(__('Last update on %s'), Html::convDateTime($this->fields["date_mod"]));
        }

        $out .= "</th>";
        $out .= "<th class='tdkb' colspan='2'>";
        if ($this->countVisibilities() == 0) {
            $out .= "<span class='red'>" . __('Unpublished') . "</span><br>";
        }

        $out .= sprintf(_n('%d view', '%d views', $this->fields["view"]), $this->fields["view"]);
        $out .= "<br>";
        if ($this->fields["is_faq"]) {
            $out .= __('This item is part of the FAQ');
        } else {
            $out .= __('This item is not part of the FAQ');
        }
        $out .= "</th></tr>";
        $out .= "</table>";

        if ($options['display']) {
            echo $out;
        } else {
            return $out;
        }

//        dlteams
        echo "<style>
            p {
                margin-top: 0;
                margin-bottom: 0.1rem !important;
            }
            
            </style>";
        return true;
    }

//    fork dlteams
    /**
     * Show visibility configuration
     *
     * @since 9.2 moved from each class to parent class
     *
     * @return void
     **/
    public function showVisibility()
    {
        /** @var array $CFG_GLPI */
        global $CFG_GLPI;

        $ID      = $this->fields['id'];
        $canedit = $this->canEdit($ID);
        $rand    = mt_rand();
        $nb      = $this->countVisibilities();
        $str_type = strtolower($this::getType());
        $fk = static::getForeignKeyField();

        if ($canedit) {
            echo "<div class='firstbloc'>";
            echo "<form name='{$str_type}visibility_form$rand' id='{$str_type}visibility_form$rand' ";
            echo " method='post' action='" . static::getFormURL() . "'>";
            echo "<input type='hidden' name='id' value='$ID'>";

            echo "<table class='tab_cadre_fixe mb-4'>";
            echo "<tr class='tab_bg_2'><th colspan='4'>" . __('Accès public / anonyme') . "</th></tr>";

            echo "<tr class='tab_bg_2'>";
//            dlteams
//            if (Session::haveRight(self::$rightname, self::PUBLISHFAQ)) {
            echo "<td width='40%'>" . __('Cet élément dispose d\'un accès publique anonyme') . "</td>";
            echo "<td>";
            Dropdown::showYesNo('is_faq', $this->fields["is_faq"]);
            echo "</td>";
            echo "<td>";
            echo "<input class='submit' type='submit' name='update' value='Enrégistrer'>";
            echo "</td>";
//            } else {
//                echo "<td>";
//                if ($this->fields["is_faq"]) {
//                    echo __('Cet élément dispose d\'un accès publique anonyme');
//                } else {
//                    echo __('Cet élément ne dispose pas d\'un accès publique', 'dlteams');
//                }
//                echo "</td>";
//            }

            echo "<td>";
            //TRANS: %d is the number of view
            if ($ID) {
                printf(_n('%d view', '%d views', $this->fields["view"]), $this->fields["view"]);
            }
            echo "</td>";
            echo "<tr>";
            echo "</table>";
            Html::closeForm();


            echo "<form name='{$str_type}visibility_form$rand' id='{$str_type}visibility_form$rand' ";
            echo " method='post' action='" . static::getFormURL() . "'>";
            echo "<input type='hidden' name='{$fk}' value='$ID'>";
            echo "<table class='tab_cadre_fixe'>";
            echo "<tr class='tab_bg_1'><th colspan='4'>" . __('Add a target') . "</tr>";
            echo "<tr class='tab_bg_1'><td class='tab_bg_2' width='100px'>";

            $types   = [
                'Entity',
                'Group',
                'Profile',
//                'User'
            ];

            $addrand = Dropdown::showItemTypes('_type', $types);
            $params = $this->getShowVisibilityDropdownParams();

            Ajax::updateItemOnSelectEvent(
                "dropdown__type" . $addrand,
                "visibility$rand",
                $CFG_GLPI["root_doc"] . "/ajax/visibility.php",
                $params
            );

            echo "</td>";
            echo "<td><span id='visibility$rand'></span>";
            echo "</td></tr>";
            echo "</table>";
            Html::closeForm();
            echo "</div>";
        }
        echo "<div class='spaced'>";
        if ($canedit && $nb) {
            Html::openMassiveActionsForm('mass' . __CLASS__ . $rand);
            $massiveactionparams = ['num_displayed'
            => min($_SESSION['glpilist_limit'], $nb),
                'container'
                => 'mass' . __CLASS__ . $rand,
                'specific_actions'
                => ['delete' => _x('button', 'Delete permanently')]
            ];

            if ($this->fields['users_id'] != Session::getLoginUserID()) {
                $massiveactionparams['confirm']
                    = __('Caution! You are not the author of this element. Delete targets can result in loss of access to that element.');
            }
            Html::showMassiveActions($massiveactionparams);
        }
        echo "<table class='tab_cadre_fixehov'>";
        $header_begin  = "<tr>";
        $header_top    = '';
        $header_bottom = '';
        $header_end    = '';
        if ($canedit && $nb) {
            $header_begin  .= "<th width='10'>";
            $header_top    .= Html::getCheckAllAsCheckbox('mass' . __CLASS__ . $rand);
            $header_bottom .= Html::getCheckAllAsCheckbox('mass' . __CLASS__ . $rand);
            $header_end    .= "</th>";
        }
        $header_end .= "<th>" . _n('Type', 'Types', 1) . "</th>";
        $header_end .= "<th>" . _n('Recipient', 'Recipients', Session::getPluralNumber()) . "</th>";
        $header_end .= "</tr>";
        echo $header_begin . $header_top . $header_end;

        // Users
        if (count($this->users)) {
            foreach ($this->users as $val) {
                foreach ($val as $data) {
                    echo "<tr class='tab_bg_1'>";
                    if ($canedit) {
                        echo "<td>";
                        Html::showMassiveActionCheckBox(PluginDlteamsDocument_User::class, $data["id"]);
                        echo "</td>";
                    }
                    echo "<td>" . User::getTypeName(1) . "</td>";
                    echo "<td>" . getUserName($data['users_id']) . "</td>";
                    echo "</tr>";
                }
            }
        }

        // Groups
        if (count($this->groups)) {
            foreach ($this->groups as $val) {
                foreach ($val as $data) {
                    echo "<tr class='tab_bg_1'>";
                    if ($canedit) {
                        echo "<td>";
                        Html::showMassiveActionCheckBox(PluginDlteamsDocument_Group::class, $data["id"]);
                        echo "</td>";
                    }
                    echo "<td>" . Group::getTypeName(1) . "</td>";

                    $names   = Dropdown::getDropdownName('glpi_groups', $data['groups_id'], 1);
                    $entname = sprintf(
                        __('%1$s %2$s'),
                        $names["name"],
                        Html::showToolTip($names["comment"], ['display' => false])
                    );
                    if ($data['entities_id'] !== null) {
                        $entname = sprintf(
                            __('%1$s / %2$s'),
                            $entname,
                            Dropdown::getDropdownName(
                                'glpi_entities',
                                $data['entities_id']
                            )
                        );
                        if ($data['is_recursive']) {
                            //TRANS: R for Recursive
                            $entname = sprintf(
                                __('%1$s %2$s'),
                                $entname,
                                "<span class='b'>(" . __('R') . ")</span>"
                            );
                        }
                    }
                    echo "<td>" . $entname . "</td>";
                    echo "</tr>";
                }
            }
        }

        // Entity
        if (count($this->entities)) {
            foreach ($this->entities as $val) {
                foreach ($val as $data) {
                    echo "<tr class='tab_bg_1'>";
                    if ($canedit) {
                        echo "<td>";
                        Html::showMassiveActionCheckBox(PluginDlteamsDocument_Entity::class, $data["id"]);
                        echo "</td>";
                    }
                    echo "<td>" . Entity::getTypeName(1) . "</td>";
                    $names   = Dropdown::getDropdownName('glpi_entities', $data['entities_id'], 1);
                    $tooltip = Html::showToolTip($names["comment"], ['display' => false]);
                    $entname = sprintf(__('%1$s %2$s'), $names["name"], $tooltip);
                    if ($data['is_recursive']) {
                        $entname = sprintf(
                            __('%1$s %2$s'),
                            $entname,
                            "<span class='b'>(" . __('R') . ")</span>"
                        );
                    }
                    echo "<td>" . $entname . "</td>";
                    echo "</tr>";
                }
            }
        }

        // Profiles
        if (count($this->profiles)) {
            foreach ($this->profiles as $val) {
                foreach ($val as $data) {
                    echo "<tr class='tab_bg_1'>";
                    if ($canedit) {
                        echo "<td>";
                        //Knowledgebase-specific case
//                        if ($this::getType() === "KnowbaseItem") {
                        Html::showMassiveActionCheckBox(PluginDlteamsDocument_Profile::class, $data["id"]);
//                        } else {
//                            Html::showMassiveActionCheckBox('Profile_' . $this::getType(), $data["id"]);
//                        }
                        echo "</td>";
                    }
                    echo "<td>" . _n('Profile', 'Profiles', 1) . "</td>";

                    $names   = Dropdown::getDropdownName('glpi_profiles', $data['profiles_id'], 1);
                    $tooltip = Html::showToolTip($names["comment"], ['display' => false]);
                    $entname = sprintf(__('%1$s %2$s'), $names["name"], $tooltip);
                    if ($data['entities_id'] !== null) {
                        $entname = sprintf(
                            __('%1$s / %2$s'),
                            $entname,
                            Dropdown::getDropdownName(
                                'glpi_entities',
                                $data['entities_id']
                            )
                        );
                        if ($data['is_recursive']) {
                            $entname = sprintf(
                                __('%1$s %2$s'),
                                $entname,
                                "<span class='b'>(" . __('R') . ")</span>"
                            );
                        }
                    }
                    echo "<td>" . $entname . "</td>";
                    echo "</tr>";
                }
            }
        }

        if ($nb) {
            echo $header_begin . $header_bottom . $header_end;
        }
        echo "</table>";
        if ($canedit && $nb) {
            $massiveactionparams['ontop'] = false;
            Html::showMassiveActions($massiveactionparams);
            Html::closeForm();
        }

        echo "</div>";
        // Add items

        return true;
    }

//    dlteams
    /**
     * Get dropdown parameters from showVisibility method
     *
     * @return array
     */
    protected function getShowVisibilityDropdownParams()
    {
        $params = parent::getShowVisibilityDropdownParams();
        $params['right'] = ($this->getField('is_faq') ? 'faq' : 'knowbase');
        $params['allusers'] = 1;
        return $params;
    }


    public static function canCreate()
    {

        // Have right to add document OR ticket followup
        return (Session::haveRight('document', CREATE)
            || Session::haveRight('followup', ITILFollowup::ADDMYTICKET));
    }


    public function canCreateItem()
    {

        if (isset($this->input['itemtype']) && isset($this->input['items_id'])) {
            if (
                ($item = getItemForItemtype($this->input['itemtype']))
                && $item->getFromDB($this->input['items_id'])
            ) {
                return $item->canAddItem('Document');
            } else {
                unset($this->input['itemtype'], $this->input['items_id']);
            }
        }

        // From Ticket Document Tab => check right to add followup.
        if (
            isset($this->fields['tickets_id'])
            && ($this->fields['tickets_id'] > 0)
        ) {
            $ticket = new Ticket();
            if ($ticket->getFromDB($this->fields['tickets_id'])) {
                return $ticket->canAddFollowups();
            }
        }

        if (Document::canCreate()) {
            return parent::canCreateItem();
        }
        return false;
    }


    public function cleanDBonPurge()
    {

        $this->deleteChildrenAndRelationsFromDb(
            [
                Document_Item::class,
            ]
        );

        // UNLINK DU FICHIER
        if (!empty($this->fields["filepath"])) {
            if (
                is_file(GLPI_DOC_DIR . "/" . $this->fields["filepath"])
                && !is_dir(GLPI_DOC_DIR . "/" . $this->fields["filepath"])
                && (countElementsInTable(
                        $this->getTable(),
                        ['sha1sum' => $this->fields["sha1sum"] ]
                    ) <= 1)
            ) {
                if (unlink(GLPI_DOC_DIR . "/" . $this->fields["filepath"])) {
                    Session::addMessageAfterRedirect(sprintf(
                        __('Successful deletion of the file %s'),
                        $this->fields["filepath"]
                    ));
                } else {
                    trigger_error(
                        sprintf(
                            'Failed to delete the file %s',
                            GLPI_DOC_DIR . "/" . $this->fields["filepath"]
                        ),
                        E_USER_WARNING
                    );
                    Session::addMessageAfterRedirect(
                        sprintf(
                            __('Failed to delete the file %s'),
                            $this->fields["filepath"]
                        ),
                        false,
                        ERROR
                    );
                }
            }
        }
    }


    public function defineTabs($options = [])
    {
        $ong = [];
//        $this->addDefaultFormTab($ong);
        $this->addStandardTab(__CLASS__, $ong, $options);
//        fork dlteams
        if($this->fields["is_html"]) {
            $this->addStandardTab(PluginDlteamsPolicieForm_Section::class, $ong, $options);
//        fork dlteams
            $this->addStandardTab(PluginDlteamsPublish::class, $ong, $options);
            $this->addStandardTab(PluginDlteamsDeliverableNotification::class, $ong, $options);
        }
//        dlteams
        // $this->addStandardTab('Document_Item', $ong, $options);
        $this->addStandardTab(PluginDlteamsObject_document::class, $ong, $options);
        $this->addImpactTab($ong, $options);
        $this->addStandardTab(PluginDlteamsTicket_Item::class, $ong, $options);
        $this->addStandardTab('Notepad', $ong, $options);
        $this->addStandardTab(PluginDlteamsMessage::class, $ong, $options);
        $this->addStandardTab('Log', $ong, $options);

        return $ong;
    }


    public function prepareInputForAdd($input)
    {
        /** @var array $CFG_GLPI */
        global $CFG_GLPI;

        // security (don't accept filename from $_REQUEST)
        if (array_key_exists('filename', $_REQUEST)) {
            unset($input['filename']);
        }

        if ($uid = Session::getLoginUserID()) {
            $input["users_id"] = Session::getLoginUserID();
        }

        // Create a doc only selecting a file from a item form
        $create_from_item = false;
        if (
            isset($input["items_id"])
            && isset($input["itemtype"])
            && ($item = getItemForItemtype($input["itemtype"]))
            && ($input["items_id"] > 0)
        ) {
            $typename = $item->getTypeName(1);
            $name     = NOT_AVAILABLE;

            if ($item->getFromDB($input["items_id"])) {
                $name = $item->getNameID();
            }
            //TRANS: %1$s is Document, %2$s is item type, %3$s is item name
            $input["name"] = addslashes(Html::resume_text(
                sprintf(
                    __('%1$s: %2$s'),
                    Document::getTypeName(1),
                    sprintf(__('%1$s - %2$s'), $typename, $name)
                ),
                200
            ));
            $create_from_item = true;
        }

        $upload_ok = false;
        if (isset($input["_filename"]) && !(empty($input["_filename"]) == 1)) {
            $upload_ok = $this->moveDocument($input, stripslashes(array_shift($input["_filename"])));
        } else if (isset($input["upload_file"]) && !empty($input["upload_file"])) {
            // Move doc from upload dir
            $upload_ok = $this->moveUploadedDocument($input, $input["upload_file"]);
        } else if (isset($input['filepath']) && file_exists(GLPI_DOC_DIR . '/' . $input['filepath'])) {
            // Document is created using an existing document file
            $upload_ok = true;
        }

        // Tag
        if (isset($input["_tag_filename"]) && !empty($input["_tag_filename"]) == 1) {
            $input['tag'] = array_shift($input["_tag_filename"]);
        }

        if (!isset($input["tag"]) || empty($input["tag"])) {
            $input['tag'] = Rule::getUuid();
        }

        // Upload failed : do not create document
        if ($create_from_item && !$upload_ok) {
            return false;
        }

        // Default document name
        if (
            (!isset($input['name']) || empty($input['name']))
            && isset($input['filename'])
        ) {
            $input['name'] = $input['filename'];
        }

        unset($input["upload_file"]);

        // Don't add if no file
        if (
            isset($input["_only_if_upload_succeed"])
            && $input["_only_if_upload_succeed"]
            && (!isset($input['filename']) || empty($input['filename']))
        ) {
            return false;
        }

        // Set default category for document linked to tickets
        if (
            isset($input['itemtype']) && ($input['itemtype'] == 'Ticket')
            && (!isset($input['documentcategories_id']) || ($input['documentcategories_id'] == 0))
        ) {
            $input['documentcategories_id'] = $CFG_GLPI["documentcategories_id_forticket"];
        }

        if (isset($input['link']) && !empty($input['link']) && !Toolbox::isValidWebUrl($input['link'])) {
            Session::addMessageAfterRedirect(
                __('Invalid link'),
                false,
                ERROR
            );
            return false;
        }

        /* Unicity check
        if (isset($input['sha1sum'])) {
          // Check if already upload in the current entity
          $crit = array('sha1sum'=>$input['sha1sum'],
                        'entities_id'=>$input['entities_id']);
          foreach ($DB->request($this->getTable(), $crit) as $data) {
             $link=$this->getFormURL();
             Session::addMessageAfterRedirect(__('"A document with that filename has already been attached to another record.').
                "&nbsp;: <a href=\"".$link."?id=".
                      $data['id']."\">".$data['name']."</a>",
                false, ERROR, true);
             return false;
          }
        } */

//        fork dlteams
        if(isset($input["documentcategories_json"]))
            $input["documentcategories_json"] = json_encode($input["documentcategories_json"]);
        return $input;
    }


    public function post_addItem()
    {

        if (
            isset($this->input["items_id"])
            && isset($this->input["itemtype"])
            && (($this->input["items_id"] > 0)
                || (($this->input["items_id"] == 0)
                    && ($this->input["itemtype"] == 'Entity')))
            && !empty($this->input["itemtype"])
        ) {
            $docitem = new Document_Item();
            $docitem->add(['documents_id' => $this->fields['id'],
                'itemtype'     => $this->input["itemtype"],
                'items_id'     => $this->input["items_id"]
            ]);

            if (is_a($this->input["itemtype"], CommonITILObject::class, true)) {
                $itilobject = new $this->input["itemtype"]();
                $itilobject->getFromDB($this->input["items_id"]);
                $this->updateParentStatus($itilobject, $this->input);
            }

            Event::log(
                $this->fields['id'],
                "documents",
                4,
                "document",
                //TRANS: %s is the user login
                sprintf(__('%s adds a link with an item'), $_SESSION["glpiname"])
            );
        }

//        dlteams
        $de = new PluginDlteamsDocument_Entity();
        $de->add([
            "documents_id" => $this->fields["id"],
            "entities_id" => Session::getActiveEntity()
        ]);

        $dp = new PluginDlteamsDocument_Profile();
        $dp->add([
            "documents_id" => $this->fields["id"],
            "entities_id" => Session::getActiveEntity(),
            "profiles_id" => $_SESSION["glpiactiveprofile"]['id']
        ]);
    }


    public function post_getFromDB()
    {
        if (
            isAPI()
            && (isset($_SERVER['HTTP_ACCEPT']) && $_SERVER['HTTP_ACCEPT'] == 'application/octet-stream'
                || isset($_GET['alt']) && $_GET['alt'] == 'media')
        ) {
            // This is a API request to download the document
            $this->send();
            exit();
        }

//        dlteams

        // Users
        $this->users    = PluginDlteamsDocument_User::getUsers($this->fields['id']);

        // Entities
        $this->entities = PluginDlteamsDocument_Entity::getEntities($this->fields['id']);

        // Group / entities
        $this->groups   = PluginDlteamsDocument_Group::getGroups($this->fields['id']);

        // Profile / entities
        $this->profiles = PluginDlteamsDocument_Profile::getProfiles($this->fields['id']);
    }


    public function prepareInputForUpdate($input)
    {

        // security (don't accept filename from $_REQUEST)
        if (array_key_exists('filename', $_REQUEST)) {
            unset($input['filename']);
        }

        if (isset($input['current_filepath'])) {
            if (isset($input["_filename"]) && !empty($input["_filename"]) == 1) {
                $this->moveDocument($input, stripslashes(array_shift($input["_filename"])));
            } else if (isset($input["upload_file"]) && !empty($input["upload_file"])) {
                // Move doc from upload dir
                $this->moveUploadedDocument($input, $input["upload_file"]);
            }
        }

        unset($input['current_filepath']);
        unset($input['current_filename']);

        if (isset($input['link']) && !empty($input['link'])  && !Toolbox::isValidWebUrl($input['link'])) {
            Session::addMessageAfterRedirect(
                __('Invalid link'),
                false,
                ERROR
            );
            return false;
        }
//        fork dlteams
        if(isset($input["documentcategories_json"]))
            $input["documentcategories_json"] = json_encode($input["documentcategories_json"]);

        return $input;
    }


    /**
     * Print the document form
     *
     * @param $ID        integer ID of the item
     * @param $options   array
     *     - target filename : where to go when done.
     *     - withtemplate boolean : template or basic item
     *
     * @return void
     **/
    public function showForm($ID, array $options = [])
    {
        $this->initForm($ID, $options);
        // $options['formoptions'] = " enctype='multipart/form-data'";
        $this->showFormHeader($options);

        $showuserlink = 0;
        if (Session::haveRight('user', READ)) {
            $showuserlink = 1;
        }

//        edited by removed.. fork dlteams

        echo "<tr class='tab_bg_1'>";
        echo "<td>" . __('Name') . "</td>";
//        dlteams
        echo "<td colspan='3'>";
        echo Html::input('name', ['value' => $this->fields['name']]);
//        dlteams
        echo "</td></tr>";
        echo "</tr>";

        echo "<tr class='tab_bg_1'>";

//        echo "<td>";
//        $showuserlink = 0;
//        if (Session::haveRight('user', READ)) {
//            $showuserlink = 1;
//        }
//        if ($this->fields["users_id"]) {
//            //TRANS: %s is the writer name
//            printf(__('%1$s: %2$s'), __('Writer'), getUserName(
//                $this->fields["users_id"],
//                $showuserlink
//            ));
//        }
//        echo "</td>";
//
//        echo "</tr>\n";

        echo "<tr class='tab_bg_1'>";
        echo "<td>" . __('Heading') . "</td>";
        echo "<td>";
        DocumentCategory::dropdown([
            'value' => $this->fields["documentcategories_id"]
        ]);
        echo "</td>";

        echo "<td>"; echo "Alias</td>"; echo "<td>";
        $params = [
            'addicon'  => DocumentCategory::canCreate(),
            'name'     => 'documentcategories_json[]',
            'right'    => 'all',
            'width'    => "300px",
            'multiple' => true,
            'values' => json_decode($this->fields["documentcategories_json"] ?? '{}'),
            'value' => null,
        ];

        DocumentCategory::dropdown($params);
        echo "</td></tr>";

        echo "<tr class='tab_bg_1'>";

//            fork dlteams ( removed block )
//        if ($ID > 0) {
//            echo "<tr><th colspan='2'>";
//            if ($this->fields["users_id"] > 0) {
//                printf(__('Added by %s'), getUserName($this->fields["users_id"], $showuserlink));
//            } else {
//                echo "&nbsp;";
//            }
//            echo "</th>";
//            echo "<th colspan='2'>";
//            echo "</th></tr>\n";
//        }

        if ($ID > 0) {
            echo "<td>" . __('Current file') . "</td>";
            echo "<td>" . $this->getDownloadLink(null, 45);
            echo "<input type='hidden' name='current_filepath' value='" . $this->fields["filepath"] . "'>";
            echo "<input type='hidden' name='current_filename' value='" . $this->fields["filename"] . "'>";
            echo "</td>";
        } else {
            echo "<td colspan=2>&nbsp;</td>";
        }

//        dlteams
        echo "</tr>";

        echo "<tr class='tab_bg_1'>";
        echo "<td>" . __('Web link') . "</td>";
        echo "<td colspan='2'>";
//        TODO: dlregister
        $link = $this->fields['link'];
        echo "<div style='display: flex; align-items: center; gap: 0.3rem'>";
        echo Html::input('link', ['value' => $link]);
        if(!empty($link))
            echo "<a href='$link' target='_blank'><i class='fa fa-external-link'></i></a>";
        echo "</div>";
        echo "</td></tr>";

        if(!$this->isNewItem()){
            echo "<tr class='tab_bg_1'>";
            echo "<td>" . __('Fichier HTML') . "</td>";
            echo "<td colspan='2'>";
            echo "<div style='display: flex; align-items: center; gap: 0.3rem'>";
            echo "<div>";

            if(!$this->fields["deliverables_id"]){
                echo '<div style="width: 1.5rem; height: 1.5rem" class="btn btn-xs btn-outline-secondary add_deliverable">'
                    . "<i class='fa-fw ti ti-plus'></i>"
                    . '</div>';
                echo "</div>";

                echo "<script>

                $('.add_deliverable').click(function(e) {
                  e.preventDefault();
                  
                  $.ajax({
                                url: '/marketplace/dlteams/ajax/add_deliverable_to_document.php',
                                type: 'POST',
                                data: {
                                    id: '".$this->fields["id"]."',
                                },
                                success: function (data) {
                                    window.location.reload();
                                }
                            });  
                });
            </script>";
            }

            if($this->fields["deliverables_id"]){
                $deliverable = new PluginDlteamsDeliverable();
                $deliverable->getFromDB($this->fields["deliverables_id"]);

//            $url = Toolbox::getItemTypeFormURL("PluginDlteamsDeliverable")
//                ."?items_id=".$this->fields["deliverables_id"]."&report_type="
//                .PluginDlteamsPublish::REPORT_SINGLE_RECORD
//                ."&print_comments=".$deliverable->fields["is_comment"]
//                ."&prevent_contextmenu=true"
//                ."&print_first_page=".$deliverable->fields["is_firstpage"]
//                ."&edit_pdf=true&itemtype=PluginDlteamsDeliverable";


                $deliverable_link = PluginDlteamsDeliverable::getFormURLWithID($this->fields['deliverables_id']);

                $url = Toolbox::getItemTypeFormURL('PluginDlteamsDeliverable')
                    . '?items_id='      . $this->fields['deliverables_id']
                    . '&report_type='    . PluginDlteamsPublish::REPORT_SINGLE_RECORD
                    . '&print_comments=' . $this->fields['is_comment']
                    . '&prevent_contextmenu=true'
                    . '&print_first_page=' . $this->fields['is_firstpage']
                    . '&edit_pdf=true'
                    . '&itemtype=PluginDlteamsDeliverable';
                echo "<div style='display: flex; gap: 1rem; align-items: center'>";
                echo "<a href='$deliverable_link' target='_blank'>Editer</a>";
                echo "<a class='btn btn-outline-secondary' title='Voir le document HTML dans un nouvel onglet' data-bs-toggle='tooltip' style='width: 1.5rem; height: 1.5rem' target='_blank' href='" . $url . "' id='btn_publication_folder'><i class='fa fa-eye'></i></a> </div>";
            }

            echo "</div>";
            echo "</td>";

            echo "<td style='position: relative'>";

            if($this->fields["deliverables_id"])
                echo "<iframe style='position: absolute; top: -160%; width: 100%; height: 600%' src='$url'></iframe>";

            echo "</td>";
            echo "</tr>";



            echo "<style>
            iframe::-webkit-scrollbar {
            width: 10px; /* Largeur de la scrollbar */
        }

        iframe::-webkit-scrollbar-track {
            background: #f4f4f4; /* Couleur du fond */
            border-radius: 10px;
        }

        iframe::-webkit-scrollbar-thumb {
            background: #888; /* Couleur de la barre */
            border-radius: 10px;
        }

        iframe::-webkit-scrollbar-thumb:hover {
            background: #555; /* Couleur au survol */
        }

        /* Pour Firefox */
        iframe {
            scrollbar-width: thin; /* Réduit la largeur de la scrollbar */
            scrollbar-color: #888 #f4f4f4; /* Couleur de la barre et du fond */
        }
        </style>";
        }

        echo "<tr class='tab_bg_1'>";
        echo "<td rowspan='1' class='middle'>" . __('Comments') . "</td>";
        echo "<td class='middle' rowspan='1'>";
        echo "<textarea class='form-control' name='comment' >" . $this->fields["comment"] . "</textarea>";
        echo "</td></tr>";

        echo "<tr class='tab_bg_1'>";
        echo "<td>" . __('MIME type') . "</td>";
        echo "<td>";
        echo Html::input('mime', ['value' => $this->fields['mime']]);
        echo "</td></tr>";

        //        dlteams
        if ($ID > 0) {
            echo "<tr class='tab_bg_1'>";
            echo "<td>" . sprintf(__('%1$s (%2$s)'), __('Checksum'), __('SHA1')) . "</td>";
            echo "<td>" . $this->fields["sha1sum"];
            echo "</td></tr>";
        }

        echo "<tr class='tab_bg_1'>";
        echo "<td>" . __('Blacklisted for import') . "</td>";
        echo "<td>";
        Dropdown::showYesNo("is_blacklisted", $this->fields["is_blacklisted"]);
        echo "</td></tr>";



//        echo "<tr class='tab_bg_1'>";
//        echo "<td>" . __('Fichier html') . "</td>";
//        echo "<td>";
//        echo '<div class="impact-side-settings-item">';
//        echo Html::getCheckbox([
//            'id'      => "is_html",
//            'name'    => "is_html",
//            'checked' => $this->fields["is_html"],
//        ]);
//        echo '</div>';
//        echo "</td></tr>";


        echo "<tr class='tab_bg_1'>";
        echo "<td>" . __('Use a FTP installed file') . "</td>";
        echo "<td>";
        $this->showUploadedFilesDropdown("upload_file");
        echo "</td>";

        echo "<td>" . sprintf(__('%1$s (%2$s)'), __('File'), self::getMaxUploadSize()) . "</td>";
        echo "<td>";
        Html::file();
        echo "</td></tr>";

        if ($ID > 0) {
            echo "<tr><th colspan='2'>";
            if ($this->fields["users_id"] > 0) {
                printf(__('Added by %s'), getUserName($this->fields["users_id"], $showuserlink));
            } else {
                echo "&nbsp;";
            }
            echo "</th>";
            echo "<th colspan='2'>";

            //TRANS: %s is the datetime of update
            printf(__('Last update on %s'), Html::convDateTime($this->fields["date_mod"]));

            echo "</th></tr>\n";
        }

        $this->showFormButtons($options);

        return true;
    }


    /**
     * Print the document form
     *
     * @param $ID        integer ID of the item
     * @param $options   array
     *     - target filename : where to go when done.
     *     - withtemplate boolean : template or basic item
     *
     * @return void
     **/
//    fork dlteams
    public function showDocumentPreview($ID, array $options = [], bool $display = true)
    {
        // On accumule tout dans $output
        $output = '';

        // Mode HTML natif
        if ($this->fields['deliverables_id']) {
            $itemtypeClass = PluginDlteamsDeliverable::class;
            /** @var PluginDlteamsDeliverable $item */
            $item = new $itemtypeClass();

            if ($item->getFromDB($this->fields['deliverables_id'])) {
                $url = Toolbox::getItemTypeFormURL('PluginDlteamsDeliverable')
                    . '?items_id='      . $this->fields['deliverables_id']
                    . '&report_type='    . PluginDlteamsPublish::REPORT_SINGLE_RECORD
                    . '&print_comments=' . $item->fields['is_comment']
                    . '&prevent_contextmenu=true'
                    . '&print_first_page=' . $item->fields['is_firstpage']
                    . '&edit_pdf=true'
                    . '&itemtype=PluginDlteamsDeliverable';

                $output .= "<iframe style='position:absolute; height:100%; width:100%' src=\"{$url}\"></iframe>";
            }

            // Affiche ou retourne
            if ($display) {
                echo $output;
                return;
            }
            return $output;
        }

        // 0) Ajustements pour supporter de gros fichiers
        ini_set('memory_limit', '512M');
        set_time_limit(300);

        // 1) CSS pour scrollbar
        $output .= <<<HTML
<style>
    iframe::-webkit-scrollbar { width: 10px; }
    iframe::-webkit-scrollbar-track { background: #f4f4f4; border-radius: 10px; }
    iframe::-webkit-scrollbar-thumb { background: #888; border-radius: 10px; }
    iframe::-webkit-scrollbar-thumb:hover { background: #555; }
    iframe { scrollbar-width: thin; scrollbar-color: #888 #f4f4f4; }
</style>
HTML;

        // 2) Chemin du fichier
        $file = GLPI_DOC_DIR . '/' . $this->fields['filepath'];
        if (!is_file($file) || !is_readable($file)) {
            $output .= '<p>Erreur : fichier introuvable ou inaccessible.</p>';
            if ($display) {
                echo $output;
                return;
            }
            return $output;
        }

        // 3) Préparation du cache
        $cacheDir = GLPI_VAR_DIR . '/cache/preview';
        if (!is_dir($cacheDir)) {
            mkdir($cacheDir, 0755, true);
        }
        foreach (glob("$cacheDir/*") as $f) {
            if (is_file($f) && filemtime($f) < time() - 86400) {
                @unlink($f);
            }
        }

        // 4) Détection de l’extension
        $ext = strtolower(pathinfo($file, PATHINFO_EXTENSION));

        // 5) Gestion selon le type
        switch ($ext) {
            // Office → PDF via LibreOffice
            case 'doc': case 'docx':
            case 'xls': case 'xlsx':
            case 'ppt': case 'pptx': case 'txt':
            if (! $this->isLibreOfficeInstalled()) {
                $output .= '<p>Erreur : LibreOffice n’est pas installé sur le serveur, impossible de prévisualiser ce document.</p>';
                break;
            }
            // handlePdfConversion doit maintenant retourner du HTML ou du chemin à inclure
            $output .= $this->handlePdfConversion($file, $cacheDir);
            break;

            // PDF natif
            case 'pdf':
                // serveOrEmbed doit retourner sa sortie
                $output .= $this->serveOrEmbed($file, 'application/pdf');
                break;

            // Images
            case 'png': case 'jpg': case 'jpeg':
            case 'gif': case 'bmp': case 'svg':
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            $mimeType = finfo_file($finfo, $file) ?: 'application/octet-stream';
            finfo_close($finfo);

            // 5.3.2 Lecture du contenu binaire et encodage en Base64
            $binaryData = file_get_contents($file);
            if ($binaryData === false) {
                $output .= '<p>Erreur : impossible de lire le fichier image.</p>';
                break;
            }
            $base64Data = base64_encode($binaryData);

            // 5.3.3 Construction de la balise <img>
            // On limite la hauteur à 600px pour rester cohérent avec vos iframes
            $output .= sprintf(
                '<div style="text-align:center; margin:1em 0;">
                    <img src="data:%s;base64,%s" style="max-width:100%%; max-height:600px; border:1px solid #ccc;" alt="Aperçu de l’image" />
                  </div>',
                $mimeType,
                $base64Data
            );
            break;

            // CSV → HTML
            case 'csv':
                $output .= $this->csvToHtml($file);
                break;

            // Fallback
            default:
                $output .= '<p>Prévisualisation non supportée pour ce type de fichier.</p>';
                break;
        }

        // Affiche ou retourne selon $display
        if ($display) {
            echo $output;
            return;
        }
        return $output;
    }


    /**
     * Vérifie la présence de la commande LibreOffice sur le serveur.
     *
     * @return bool
     */
    private function isLibreOfficeInstalled(): bool {
        exec('which libreoffice', $out, $ret);
        return ($ret === 0);
    }

    /**
     * Convertit un document Office en PDF (cache + conversion si nécessaire),
     * puis retourne soit le code HTML d'un iframe, soit le contenu brut du PDF
     * selon l'usage de serveOrEmbed().
     *
     * @param string $file     Chemin absolu du fichier source
     * @param string $cacheDir Dossier de cache pour les PDF générés
     * @return string          HTML à intégrer dans la page
     */
    private function handlePdfConversion(string $file, string $cacheDir): string
    {
        $output       = '';
        $cacheName    = sha1($file) . '.pdf';
        $cachedPdf    = $cacheDir . '/' . $cacheName;

        // Conversion si nécessaire
        if (!file_exists($cachedPdf) || filemtime($cachedPdf) < filemtime($file)) {
            $cmd = sprintf(
                'libreoffice --headless --convert-to pdf --outdir %s %s > /dev/null 2>&1',
                escapeshellarg($cacheDir),
                escapeshellarg($file)
            );
            exec($cmd, $out, $ret);

            $origPdf = $cacheDir . '/' . pathinfo($file, PATHINFO_FILENAME) . '.pdf';
            if ($ret !== 0 || !file_exists($origPdf)) {
                // Erreur de conversion
                return '<p>Erreur lors de la prévisualisation du document.</p>';
            }

            // On renomme et on nettoie l'original
            @rename($origPdf, $cachedPdf);
        }

        // Si tout est OK, on renvoie le HTML d'affichage
        $output .= $this->serveOrEmbed($cachedPdf, 'application/pdf');
        return $output;
    }


    /**
     * Si preview=1, sert le fichier brut avec le bon header et quitte le script.
     * Sinon, retourne le HTML d'un <iframe> pointant vers la même URL + preview=1.
     *
     * @param  string $path Chemin absolu du fichier
     * @param  string $mime MIME type à envoyer
     * @return string       Code HTML de l'iframe (quand preview≠1)
     */
    private function serveOrEmbed(string $path, string $mime): string
    {
        // Cas de la requête directe : on renvoie le PDF brut
        if (isset($_GET['preview']) && $_GET['preview'] === '1') {
            header('Content-Type: ' . $mime);
            readfile($path);
            exit;
        }

        // Sinon on construit l'URL avec le paramètre preview=1
        $delimiter = (strpos($_SERVER['REQUEST_URI'], '?') !== false) ? '&' : '?';
        $url       = htmlspecialchars($_SERVER['REQUEST_URI'] . $delimiter . 'preview=1');

        // On retourne le code HTML de l'iframe
        return sprintf(
            '<iframe src="%s" width="100%%" height="900px" style="border:1px solid #ccc;"></iframe>',
            $url
        );
    }


    /**
     * Convertit un CSV en tableau HTML avec <thead> et <tbody>.
     *
     * @param string $file Chemin absolu du fichier CSV
     * @return string
     */
    private function csvToHtml(string $file): string {
        $html = '
        <style>
          .csv-preview { border-collapse: collapse; width: 100%; }
          .csv-preview th, .csv-preview td {
            border: 1px solid #ccc;
            padding: 6px 8px;
            text-align: left;
          }
          .csv-preview th {
            background-color: #f9f9f9;
          }
          .csv-container { overflow: auto; max-height: 600px; margin-bottom: 1em; }
        </style>
        <div class="csv-container">
          <table class="csv-preview">';

        if (($handle = fopen($file, 'r')) !== false) {
            $firstRow = true;
            while (($row = fgetcsv($handle, 0, ',')) !== false) {
                if ($firstRow) {
                    $html .= '<thead><tr>';
                    foreach ($row as $cell) {
                        $html .= '<th>' . htmlspecialchars($cell) . '</th>';
                    }
                    $html .= '</tr></thead><tbody>';
                    $firstRow = false;
                } else {
                    $html .= '<tr>';
                    foreach ($row as $cell) {
                        $html .= '<td>' . htmlspecialchars($cell) . '</td>';
                    }
                    $html .= '</tr>';
                }
            }
            fclose($handle);
            $html .= '</tbody>';
        } else {
            $html .= '<tr><td colspan="100%">Impossible de lire le fichier CSV.</td></tr>';
        }

        $html .= '
          </table>
        </div>';

        return $html;
    }
//    end fork dlteams


    /**
     * Get max upload size from php config
     **/
    public static function getMaxUploadSize()
    {
        /** @var array $CFG_GLPI */
        global $CFG_GLPI;

        //TRANS: %s is a size
        return sprintf(__('%s Mio max'), $CFG_GLPI['document_max_size']);
    }


    /**
     * Send a document to navigator
     *
     * @param string $context Context to resize image, if any
     **/
    public function send($context = null, $download = false)
    {
        $file = GLPI_DOC_DIR . "/" . $this->fields['filepath'];

        $type = static::detectFileType($file);

        if ($context !== null) {
            $file = self::getImage($file, $context);
        }

        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mimeType = $finfo->file($file);


        if($mimeType == "application/pdf"){
            Toolbox::sendFile(preg_replace('/\.[^.]+$/', '.PDF', $file), $this->fields['filename'], $download?$this->fields['mime']:'application/pdf');
        }

        elseif($type == "document"){
            $pdf_filepath = preg_replace('/\.[^.]+$/', '.pdf', $file);
//            si le document equivalent pdf n'existe pas, le créer
            if(!file_exists($pdf_filepath)){
                static::convertWordToPDF($file, $pdf_filepath);
            }
            Toolbox::sendFile(preg_replace('/\.[^.]+$/', '.pdf', $file), $this->fields['filename'], $download?$this->fields['mime']:'application/pdf');
        }
        else
            Toolbox::sendFile($file, $this->fields['filename'], isset($_GET["preventdownload"])?'application/pdf':$this->fields['mime']); // fork dlteams

    }


    public static function detectFileType($filePath) {
        if (!file_exists($filePath)) {
            return "Erreur : Le fichier n'existe pas.";
        }

        // Initialiser finfo avec FILEINFO_MIME_TYPE
        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mimeType = $finfo->file($filePath);

        // Vérifier le type MIME et retourner une description simplifiée
        switch ($mimeType) {
            // Images
            case 'image/jpeg':
            case 'image/png':
            case 'image/gif':
            case 'image/svg+xml':
            case 'image/bmp':
            case 'image/tiff':
                return "image";

            // Documents
            case 'application/pdf':
            case 'application/msword':
            case 'application/vnd.openxmlformats-officedocument.wordprocessingml.document': // .docx
            case 'application/vnd.ms-excel':
            case 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet': // .xlsx
            case 'application/vnd.ms-powerpoint':
            case 'application/vnd.openxmlformats-officedocument.presentationml.presentation': // .pptx
            case 'application/vnd.oasis.opendocument.text': // .odt
            case 'application/vnd.oasis.opendocument.spreadsheet': // .ods
            case 'text/plain': // .txt
            case 'application/json': // .json
            case 'application/xml': // .xml
                return "document";

            // Vidéos
            case 'video/mp4':
            case 'video/x-msvideo': // .avi
            case 'video/mpeg':
            case 'video/x-matroska': // .mkv
            case 'video/webm':
                return "video";

            // Audio
            case 'audio/mpeg': // .mp3
            case 'audio/ogg': // .ogg
            case 'audio/wav': // .wav
            case 'audio/aac': // .aac
                return "audio";

            // Archives
            case 'application/zip': // .zip
            case 'application/x-rar-compressed': // .rar
            case 'application/x-tar': // .tar
            case 'application/x-7z-compressed': // .7z
            case 'application/gzip': // .gz
                return "archive";

            // Types MIME non reconnus
            default:
                return null;
        }
    }



    /**
     * Get download link for a document
     *
     * @param CommonDBTM|null   $linked_item    Item linked to the document, to check access right
     * @param integer           $len            maximum length of displayed string (default 20)
     *
     **/
    public function getDownloadLink($linked_item = null, $len = 20)
    {
        /**
         * @var array $CFG_GLPI
         * @var \DBmysql $DB
         */
        global $CFG_GLPI, $DB;

        $link_params = '';
        if (is_string($linked_item)) {
            // Old behaviour.
            Toolbox::deprecated('Passing additionnal URL parameters in Document::getDownloadLink() is deprecated.', true, '11.0');
            $linked_item = null;
            $link_params = $linked_item;
        } elseif ($linked_item !== null && !($linked_item instanceof CommonDBTM)) {
            throw new \InvalidArgumentException();
        } elseif ($linked_item !== null) {
            $link_params = sprintf('&itemtype=%s&items_id=%s', $linked_item->getType(), $linked_item->getID());
        }

        $splitter = $this->fields['filename'] !== null ? explode("/", $this->fields['filename']) : [];

        if (count($splitter) == 2) {
            // Old documents in EXT/filename
            $fileout = $splitter[1];
        } else {
            // New document
            $fileout = $this->fields['filename'];
        }

        $initfileout = $fileout;

        if ($fileout !== null && Toolbox::strlen($fileout) > $len) {
            $fileout = Toolbox::substr($fileout, 0, $len) . "&hellip;";
        }

        $out   = '';
        $open  = '';
        $close = '';

//        dlteams
        $opendownload = '';

        $can_view_options = $linked_item !== null
            ? ['itemtype' => $linked_item->getType(), 'items_id' => $linked_item->getID()]
            : ['itemtype' => Ticket::getType(), 'items_id' => $this->fields['tickets_id']];

        if (self::canView() || $this->canViewFile($can_view_options)) {
            $open  = "<a href='" . $CFG_GLPI["root_doc"] . "/front/document.send.php?docid=" .
                $this->fields['id'] . $link_params . "' alt=\"" . $initfileout . "\"
                    title=\"" . $initfileout . "\"target='_blank'>";
            $close = "</a>";


            $file = GLPI_DOC_DIR . "/" . $this->fields['filepath'];

            $type = static::detectFileType($file);
//            dlteams
            if($type == "document"){
                $downloadicon = '&nbsp;<svg width="20px" height="20px" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <g id="Interface / Download">
                                    <path id="Vector" d="M6 21H18M12 3V17M12 17L17 12M12 17L7 12" stroke="#000000" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                                    </g>
                                </svg>';
                $opendownload = "<a href='" . $CFG_GLPI["root_doc"] . "/front/document.send.php?docid=" .
                    $this->fields['id'] . "&download=true" . "' alt=\"" . $initfileout . "\"
                        title='Télécharger le document " . $this->fields['name'] . "' target='_blank'>".$downloadicon."</a>";
            }


        }

        $splitter = $this->fields['filename'] !== null ? explode("/", $this->fields['filepath']) : [];

        if (count($splitter)) {
            $iterator = $DB->request([
                'SELECT' => 'icon',
                'FROM'   => 'glpi_documenttypes',
                'WHERE'  => [
                    'ext'    => ['LIKE', $splitter[0]],
                    'icon'   => ['<>', '']
                ]
            ]);

            if (count($iterator) > 0) {
                $result = $iterator->current();
                $icon = $result['icon'];
                if (!file_exists(GLPI_ROOT . "/pics/icones/$icon")) {
                    $icon = "defaut-dist.png";
                }
                $out .= "&nbsp;<img class='middle' style='margin-left:3px; margin-right:6px;' alt=\"" .
                    $initfileout . "\" title=\"" . $initfileout . "\" src='" .
                    $CFG_GLPI["typedoc_icon_dir"] . "/$icon'>";
            }
        }
        $out .= "$open<span class='b'>$fileout</span>$close $opendownload";

        $urlPdf = $CFG_GLPI['root_doc']
            . "/front/document.send.php?docid="
            . $this->fields['id']."&preventdownload=true";

        if($this->fields['filepath']) {
            // Détecter le type de fichier
            $file_extension = strtolower(pathinfo($this->fields['filename'], PATHINFO_EXTENSION));
            $mime_type = $this->fields['mime'] ?? '';

            // Déterminer le contenu du tooltip selon le type
            if (in_array($file_extension, ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp', 'svg']) ||
                strpos($mime_type, 'image/') === 0) {
                // Pour les images
                $tooltip_content = '<div style="max-width: 600px; max-height: 400px; text-align: center;">
            <img src="' . $urlPdf . '" style="max-width: 100%; max-height: 100%; border: 1px solid #ccc; border-radius: 4px;" alt="Aperçu de l\'image">
        </div>';
                $tooltip_title = 'Voir l\'image ' . $this->fields['name'];
            } elseif (in_array($file_extension, ['pdf']) || strpos($mime_type, 'application/pdf') === 0) {
                // Pour les PDF
                $tooltip_content = '<iframe src="' . $urlPdf . '" width="100%" height="600px" style="border:1px solid #ccc;"></iframe>';
                $tooltip_title = 'Ouvrir un aperçu du document PDF ' . $this->fields['name'];
            } elseif (in_array($file_extension, ['txt', 'csv', 'log']) || strpos($mime_type, 'text/') === 0) {
                // Pour les fichiers texte
                $tooltip_content = '<div style="width: 600px; height: 400px; overflow: auto; border: 1px solid #ccc; padding: 10px; background: #f8f9fa; font-family: monospace; font-size: 12px;">
            <iframe src="' . $urlPdf . '" width="100%" height="100%" style="border: none;"></iframe>
        </div>';
                $tooltip_title = 'Lire le fichier texte ' . $this->fields['name'];
            } elseif (in_array($file_extension, ['html', 'htm'])) {
                // Pour les fichiers HTML
                $tooltip_content = '<div style="width: 800px; height: 600px; border: 1px solid #ccc; border-radius: 4px;">
            <iframe src="' . $urlPdf . '" width="100%" height="100%" style="border: none;"></iframe>
        </div>';
                $tooltip_title = 'Voir le contenu HTML ' . $this->fields['name'];
            } else {
                // Pour les autres types de fichiers - garder l'aperçu comme avant
                $tooltip_content = '<iframe src="' . $urlPdf . '" width="100%" height="600px" style="border:1px solid #ccc;"></iframe>';
                $tooltip_title = 'Ouvrir un aperçu du document ' . $this->fields['name'];
            }

            $out = $out . Html::showToolTip(
                    $tooltip_content,
                    [
                        'display'       => false,
                        'awesome-class' => 'fa-eye',
                        'autoclose'     => false,
                        'onclick'       => true,
                        'title'         => $tooltip_title,
                    ]
                );
        }

//        $out = htmlspecialchars_decode($out);


        return $out;
    }


    /**
     * find a document with a file attached
     *
     * @param integer $entity    entity of the document
     * @param string  $path      path of the searched file
     *
     * @return boolean
     **/
    public function getFromDBbyContent($entity, $path)
    {

        /** @var \DBmysql $DB */
        global $DB;

        if (empty($path)) {
            return false;
        }

        $sum = sha1_file($path);
        if (!$sum) {
            return false;
        }

        $doc_iterator = $DB->request(
            [
                'SELECT' => 'id',
                'FROM'   => $this->getTable(),
                'WHERE'  => [
                    $this->getTable() . '.sha1sum'      => $sum,
                    $this->getTable() . '.entities_id'  => $entity
                ],
                'LIMIT'  => 1,
            ]
        );

        if ($doc_iterator->count() === 0) {
            return false;
        }

        $doc_data = $doc_iterator->current();
        return $this->getFromDB($doc_data['id']);
    }


    /**
     * Check is the curent user is allowed to see the file.
     *
     * @param array $options array of possible options used to check rights:
     *     - itemtype/items_id:     itemtype and ID of item linked to document
     *     - changes_id (legacy):   ID of Change linked to document. Ignored if itemtype/items_id options are set.
     *     - problems_id (legacy):  ID of Problem linked to document. Ignored if itemtype/items_id options are set.
     *     - tickets_id (legacy):   ID of Ticket linked to document. Ignored if itemtype/items_id options are set.
     *
     * @return boolean
     **/
    public function canViewFile(array $options = [])
    {

        // Check if it is my doc
        if (
            Session::getLoginUserID()
            && ($this->can($this->fields["id"], READ)
                || ($this->fields["users_id"] === Session::getLoginUserID()))
        ) {
            return true;
        }

        if ($this->canViewFileFromReminder()) {
            return true;
        }

        if ($this->canViewFileFromKnowbaseItem()) {
            return true;
        }

        // new options
        $itemtype = $options['itemtype'] ?? null;
        $items_id = $options['items_id'] ?? null;

        // legacy options
        $changes_id  = $itemtype === null ? ($options['changes_id'] ?? null) : ($itemtype === 'Change' ? $items_id : null);
        $problems_id = $itemtype === null ? ($options['problems_id'] ?? null) : ($itemtype === 'Problem' ? $items_id : null);
        $tickets_id  = $itemtype === null ? ($options['tickets_id'] ?? null) : ($itemtype === 'Ticket' ? $items_id : null);

        if ($changes_id !== null && $this->canViewFileFromItilObject('Change', $changes_id)) {
            return true;
        }

        if ($problems_id !== null && $this->canViewFileFromItilObject('Problem', $problems_id)) {
            return true;
        }

        if (
            $itemtype !== null
            && is_a($itemtype, CommonDBTM::class, true)
            && $items_id !== null
            && $this->canViewFileFromItem($itemtype, $items_id)
        ) {
            return true;
        }

        // The following case should be reachable from the API
        self::loadAPISessionIfExist();

        if ($tickets_id !== null && $this->canViewFileFromItilObject('Ticket', $tickets_id)) {
            return true;
        }

        return false;
    }

    /**
     * Try to load the session from the API Tolen
     *
     * @since 9.5
     */
    private static function loadAPISessionIfExist()
    {
        $session_token = \Toolbox::getHeader('Session-Token');

        // No api token found
        if ($session_token === null) {
            return;
        }

        $current_session = session_id();

        // Clean current session
        if (!empty($current_session) && $current_session !== $session_token) {
            session_destroy();
        }

        // Load API session
        session_id($session_token);
        Session::start();
    }

    /**
     * Check if file of current instance can be viewed from a Reminder.
     *
     * @global DBmysql $DB
     * @return boolean
     *
     * @TODO Use DBmysqlIterator instead of raw SQL
     */
    private function canViewFileFromReminder()
    {

        /** @var \DBmysql $DB */
        global $DB;

        if (!Session::getLoginUserID()) {
            return false;
        }

        $criteria = array_merge_recursive(
            [
                'COUNT'     => 'cpt',
                'FROM'      => 'glpi_documents_items',
                'LEFT JOIN' => [
                    'glpi_reminders'  => [
                        'ON' => [
                            'glpi_documents_items'  => 'items_id',
                            'glpi_reminders'        => 'id', [
                                'AND' => [
                                    'glpi_documents_items.itemtype'  => 'Reminder'
                                ]
                            ]
                        ]
                    ]
                ],
                'WHERE'     => [
                    'glpi_documents_items.documents_id' => $this->fields['id']
                ]
            ],
            Reminder::getVisibilityCriteria()
        );

        $result = $DB->request($criteria)->current();
        return $result['cpt'] > 0;
    }

    /**
     * Check if file of current instance can be viewed from a KnowbaseItem.
     *
     * @global array $CFG_GLPI
     * @global DBmysql $DB
     * @return boolean
     */
    private function canViewFileFromKnowbaseItem()
    {

        /**
         * @var array $CFG_GLPI
         * @var \DBmysql $DB
         */
        global $CFG_GLPI, $DB;

        // Knowbase items can be viewed by non connected user in case of public FAQ
        if (!Session::getLoginUserID() && !$CFG_GLPI['use_public_faq']) {
            return false;
        }

        if (
            !Session::haveRight(KnowbaseItem::$rightname, READ)
            && !Session::haveRight(KnowbaseItem::$rightname, KnowbaseItem::READFAQ)
            && !$CFG_GLPI['use_public_faq']
        ) {
            return false;
        }

        $visibilityCriteria = KnowbaseItem::getVisibilityCriteria();

        $request = [
            'FROM'      => 'glpi_documents_items',
            'COUNT'     => 'cpt',
            'LEFT JOIN' => [
                'glpi_knowbaseitems' => [
                    'FKEY' => [
                        'glpi_knowbaseitems'   => 'id',
                        'glpi_documents_items' => 'items_id',
                        ['AND' => ['glpi_documents_items.itemtype' => 'KnowbaseItem']]
                    ]
                ]
            ],
            'WHERE'     => [
                'glpi_documents_items.documents_id' => $this->fields['id'],
            ]
        ];

        if (array_key_exists('LEFT JOIN', $visibilityCriteria) && count($visibilityCriteria['LEFT JOIN']) > 0) {
            $request['LEFT JOIN'] += $visibilityCriteria['LEFT JOIN'];
        }
        if (array_key_exists('WHERE', $visibilityCriteria) && count($visibilityCriteria['WHERE']) > 0) {
            $request['WHERE'] += $visibilityCriteria['WHERE'];
        }

        $result = $DB->request($request)->current();

        return $result['cpt'] > 0;
    }

    /**
     * Check if file of current instance can be viewed from a CommonITILObject.
     *
     * @global DBmysql $DB
     * @param string  $itemtype
     * @param integer $items_id
     * @return boolean
     */
    private function canViewFileFromItilObject($itemtype, $items_id)
    {

        /** @var \DBmysql $DB */
        global $DB;

        if (!Session::getLoginUserID()) {
            return false;
        }

        /* @var CommonITILObject $itil */
        $itil = new $itemtype();

        if (!$itil->can($items_id, READ)) {
            return false;
        }

        $itil->getFromDB($items_id);

        $result = $DB->request([
            'FROM'  => Document_Item::getTable(),
            'COUNT' => 'cpt',
            'WHERE' => [
                $itil->getAssociatedDocumentsCriteria(),
                'documents_id' => $this->fields['id']
            ]
        ])->current();

        return $result['cpt'] > 0;
    }

    /**
     * Check if file of current instance can be viewed from item having given itemtype/items_id.
     *
     * @global DBmysql $DB
     *
     * @param string  $itemtype
     * @param integer $items_id
     *
     * @return boolean
     */
    private function canViewFileFromItem($itemtype, $items_id): bool
    {
        /** @var \DBmysql $DB */
        global $DB;

        $item = new $itemtype();

        if (!$item->can($items_id, READ)) {
            return false;
        }

        /** @var CommonDBTM $item */
        $item->getFromDB($items_id);
        if (!$item->canViewItem()) {
            return false;
        }

        $result = $DB->request(
            [
                'FROM'  => Document_Item::getTable(),
                'COUNT' => 'cpt',
                'WHERE' => [
                    'itemtype' => $itemtype,
                    'items_id' => $items_id,
                ]
            ]
        )->current();

        if ($result['cpt'] === 0) {
            return false;
        }

        return true;
    }

    // fork dlteams // attention l'ajout de Criteria avec ET empeche l'affichage de l'arborescence
    public static function getDefaultSearchRequest()
    {
        $search = [
            'sort' => 19,
            'order' => 'DESC'
        ];
        return $search;
    }

    public static function rawSearchOptionsToAdd($itemtype = null)
    {
        $tab = [];

        $tab[] = [
            'id'                 => 'document',
            'name'               => self::getTypeName(Session::getPluralNumber())
        ];

        $tab[] = [
            'id'                 => '119',
            'table'              => 'glpi_documents_items',
            'field'              => 'id',
            'name'               => _x('quantity', 'Number of documents'),
            'forcegroupby'       => true,
            'usehaving'          => true,
            'datatype'           => 'count',
            'massiveaction'      => false,
            'joinparams'         => [
                'jointype'           => 'itemtype_item'
            ]
        ];

        return $tab;
    }


    public function rawSearchOptions()
    {
        $tab = [];

        $tab[] = [
            'id'                 => 'common',
            'name'               => __('Characteristics')
        ];

        $tab[] = [
            'id'                 => '1',
            'table'              => $this->getTable(),
            'field'              => 'name',
            'name'               => __('Name'),
            'datatype'           => 'itemlink',
            'massiveaction'      => false,
        ];

        $tab[] = [
            'id'                 => '2',
            'table'              => $this->getTable(),
            'field'              => 'id',
            'name'               => __('ID'),
            'massiveaction'      => false,
            'datatype'           => 'number'
        ];

        $tab[] = [
            'id'                 => '3',
            'table'              => $this->getTable(),
            'field'              => 'filename',
            'name'               => __('File'),
            'massiveaction'      => false,
            'datatype'           => 'string'
        ];

//        todo dlregister
        $tab[] = [
            'id'                 => '4',
            'table'              => $this->getTable(),
            'field'              => 'link',
            'name'               => __('Web link'),
            'datatype'           => 'specific',
        ];

        $tab[] = [
            'id'                 => '5',
            'table'              => $this->getTable(),
            'field'              => 'mime',
            'name'               => __('MIME type'),
            'datatype'           => 'string',
        ];

        $tab[] = [
            'id'                 => '6',
            'table'              => $this->getTable(),
            'field'              => 'tag',
            'name'               => __('Tag'),
            'datatype'           => 'text',
            'massiveaction'      => false
        ];


        $tab[] = [
            'id'                 => '7',
            'table'              => 'glpi_documentcategories',
            'field'              => 'completename',
            'name'               => __('Heading'),
            'datatype'           => 'dropdown'
        ];

//        $tab[] = [
//            'id'                 => '8',
//            'table'              => PluginDlteamsDeliverable::getTable(),
//            'field'              => 'name',
//            'linkfield'		     => 'deliverables_id',
//            'name'               => PluginDlteamsDeliverable::getTypeName(1),
//            'massiveaction'      => false,
//            'datatype'           => 'dropdown'
//        ];


        $tab[] = [
            'id'                 => '80',
            'table'              => 'glpi_entities',
            'field'              => 'completename',
            'name'               => Entity::getTypeName(1),
            'massiveaction'      => false,
            'datatype'           => 'text'
        ];

        $tab[] = [
            'id'                 => '86',
            'table'              => $this->getTable(),
            'field'              => 'is_recursive',
            'name'               => __('Child entities'),
            'datatype'           => 'bool'
        ];

        $tab[] = [
            'id'                 => '19',
            'table'              => $this->getTable(),
            'field'              => 'date_mod',
            'name'               => __('Last update'),
            'datatype'           => 'datetime',
            'massiveaction'      => false
        ];

        $tab[] = [
            'id'                 => '121',
            'table'              => $this->getTable(),
            'field'              => 'date_creation',
            'name'               => __('Creation date'),
            'datatype'           => 'datetime',
            'massiveaction'      => false
        ];

        $tab[] = [
            'id'                 => '20',
            'table'              => $this->getTable(),
            'field'              => 'sha1sum',
            'name'               => sprintf(__('%1$s (%2$s)'), __('Checksum'), __('SHA1')),
            'massiveaction'      => false,
            'datatype'           => 'string'
        ];

        $tab[] = [
            'id'                 => '16',
            'table'              => $this->getTable(),
            'field'              => 'comment',
            'name'               => __('Comments'),
            'datatype'           => 'text'
        ];

        $tab[] = [
            'id'                 => '72',
            'table'              => 'glpi_documents_items',
            'field'              => 'id',
            'name'               => _x('quantity', 'Number of associated items'),
            'forcegroupby'       => true,
            'usehaving'          => true,
            'datatype'           => 'count',
            'massiveaction'      => false,
            'joinparams'         => [
                'jointype'           => 'child'
            ]
        ];

        $tab[] = [
            'id'                 => '73',
            'table'              => $this->getTable(),
            'field'              => 'id',
            'name'               => __("Vue HTML"),
            'datatype'           => 'specific',
            'toview'             => true,
            'massiveaction'      => true,
            'htmlview'           => true
        ];

        $tab[] = [
            'id'                 => '8',
            'table'              => $this->getTable(),
            'field'              => 'id',
            'name'               => __("Editeur HTML"),
            'datatype'           => 'specific',
            'toview'             => true,
            'massiveaction'      => true,
            'htmleditor'           => true
        ];

        $tab[] = [
            'id'                 => '108',
            'table'              => $this->getTable(),
            'field'              => 'id',
            'name'               => __("HTML"),
            'datatype'           => 'specific',
            'toview'             => true,
            'massiveaction'      => true,
            'htmlshow'           => true
        ];


        $tab[] = [
            'id' => '103',
            'table' => 'glpi_plugin_dlteams_policieforms_items',
            'field' => 'id',
            'name' => _x('quantity', 'Jeux de données'),
            'forcegroupby' => true,
            'usehaving' => true,
            'datatype' => 'count',
            'massiveaction' => false,
            'joinparams' => [
                'jointype' => 'itemtype_item'
            ],
        ];

        $tab[] = [
            'id' => '104',
            'table' => 'glpi_plugin_dlteams_records_items',
            'field' => 'id',
            'name' => _x('quantity', 'Traitements'),
            'forcegroupby' => true,
            'usehaving' => true,
            'datatype' => 'count',
            'massiveaction' => false,
            'joinparams' => [
                'jointype' => 'itemtype_item'
            ],
        ];



        // add objectlock search options
        $tab = array_merge($tab, ObjectLock::rawSearchOptionsToAdd(get_class($this)));

        $tab = array_merge($tab, Notepad::rawSearchOptionsToAdd());

        return $tab;
    }

//    todo dlregister
    public static function getSpecificValueToDisplay($field, $values, array $options = [])
    {

        /*                highlight_string("<?php\n\$data =\n" . var_export($options, true) . ";\n?>");*/
//        die();
        if($options["searchopt"]["field"] == "link"){
            $link = $values["link"];
            if(!empty($link))
                return "<a href='$link' title='$link' target='_blank'>...</a>";
        }
        if(isset($options["searchopt"]["htmlview"])){

            $output = "&nbsp;";
            $dc = new Document();
            $dc->getFromDB($values["id"]);
            if($dc->fields["deliverables_id"]){
                $deliverables_id = $dc->fields["deliverables_id"];
                $output = "";
                $deliverable = new PluginDlteamsDeliverable();
                $deliverable->getFromDB($deliverables_id);

                $url = Toolbox::getItemTypeFormURL("PluginDlteamsDeliverable")
                    . "?items_id=" . $deliverables_id . "&report_type="
                    . PluginDlteamsPublish::REPORT_SINGLE_RECORD
                    . "&print_comments=true"
                    . "&prevent_contextmenu=true"
                    . "&print_first_page=true"
                    . "&edit_pdf=true&itemtype=PluginDlteamsDeliverable";

                $output.="<a class='btn btn-outline-secondary' title='Voir le document HTML dans un nouvel onglet' data-bs-toggle='tooltip' style='width: 1.5rem; height: 1.5rem' target='_blank' href='" . $url . "' id='btn_publication_folder'><i class='fa fa-eye'></i></a> </div>";
            }




            return $output;
        }

        if(isset($options["searchopt"]["htmleditor"])){

            $output = "&nbsp;";
            $dc = new Document();
            $dc->getFromDB($values["id"]);
            if($dc->fields["deliverables_id"]){
                $deliverables_id = $dc->fields["deliverables_id"];
                $output = "";

                $url = PluginDlteamsDeliverable::getFormURLWithID($deliverables_id);
                $output .= <<<HTML
                <a title="Aller sur l'éditeur HTML" data-bs-toggle="tooltip" target="_blank" href="{$url}" id="btn_publication_folder">Voir</a>
                HTML;
            }




            return $output;
        }


        if(isset($options["searchopt"]["htmlshow"])) {
            $output = "&nbsp;";
            $dc = new Document();
            $dc->getFromDB($values["id"]);
            if ($dc->fields["deliverables_id"]) {
                $deliverables_id = $dc->fields["deliverables_id"];
                $output = "";
                $deliverable = new PluginDlteamsDeliverable();
                $deliverable->getFromDB($deliverables_id);

                // URL pour voir le document HTML
                $view_url = Toolbox::getItemTypeFormURL("PluginDlteamsDeliverable")
                    . "?items_id=" . $deliverables_id . "&report_type="
                    . PluginDlteamsPublish::REPORT_SINGLE_RECORD
                    . "&print_comments=true"
                    . "&prevent_contextmenu=true"
                    . "&print_first_page=true"
                    . "&edit_pdf=true&itemtype=PluginDlteamsDeliverable";

                // URL pour éditer le document
                $edit_url = PluginDlteamsDeliverable::getFormURLWithID($deliverables_id);

                // Icône œil avec tooltip personnalisé
                $eye_tooltip = Html::showToolTip(
                    '<div style="width: 800px; height: 600px;"><iframe src="' . $view_url . '" width="100%" height="100%" style="border:1px solid #ccc; border-radius: 4px;"></iframe></div>',
                    [
                        'display'       => false,
                        'awesome-class' => 'fa-eye',
                        'autoclose'     => false,
                        'onclick'       => true,
                        'title'         => 'Aperçu du document HTML',
                    ]
                );

                // Icône stylo pour éditer
                $edit_link = "<a href='{$edit_url}' target='_blank' title='Éditer le document' style='margin-left: 5px; color: #6c757d; text-decoration: none;' onmouseover='this.style.color=\"#28a745\"' onmouseout='this.style.color=\"#6c757d\"'><i class='fa fa-pencil'></i></a>";

                $output = "<span style='display: inline-flex; align-items: center; gap: 5px;'>" . $eye_tooltip . $edit_link . "</span>";
            }

            return $output;
        }




        return parent::getSpecificValueToDisplay($field, $values, $options);
    }

    /**
     * Move a file to a new location
     * Work even if dest file already exists
     *
     * @param string $srce   source file path
     * @param string $dest   destination file path
     *
     * @return boolean : success
     **/
    public static function renameForce($srce, $dest)
    {

        // File already present
        if (is_file($dest)) {
            // As content is the same (sha1sum), no need to copy
            @unlink($srce);
            return true;
        }
        // Move
        return rename($srce, $dest);
    }


    /**
     * Move an uploadd document (files in GLPI_DOC_DIR."/_uploads" dir)
     *
     * @param array  $input     array of datas used in adding process (need current_filepath)
     * @param string $filename  filename to move
     *
     * @return boolean for success / $input array is updated
     **/
    public function moveUploadedDocument(array &$input, $filename)
    {
        if (str_contains($filename, '/') || str_contains($filename, '\\')) {
            // Filename is not supposed to contains directory separators.
            trigger_error(sprintf('Moving file `%s` is forbidden for security reasons.', $filename), E_USER_WARNING);
            return false;
        }

        $prefix = '';
        if (isset($input['_prefix_filename'])) {
            $prefix = array_shift($input['_prefix_filename']);
        }

        $fullpath = GLPI_UPLOAD_DIR . "/" . $filename;
        $filename = str_replace($prefix, '', $filename);

        if (!is_dir(GLPI_UPLOAD_DIR)) {
            Session::addMessageAfterRedirect(__("Upload directory doesn't exist"), false, ERROR);
            return false;
        }

        if (!is_file($fullpath)) {
            trigger_error(
                sprintf('File %s not found.', $fullpath),
                E_USER_WARNING
            );
            Session::addMessageAfterRedirect(
                sprintf(__('File %s not found.'), $filename),
                false,
                ERROR
            );
            return false;
        }
        $sha1sum  = sha1_file($fullpath);
        $dir      = self::isValidDoc($filename);
        $new_path = self::getUploadFileValidLocationName($dir, $sha1sum);

        if (!$sha1sum || !$dir || !$new_path) {
            return false;
        }

        // Delete old file (if not used by another doc)
        if (
            isset($input['current_filepath'])
            && !empty($input['current_filepath'])
            && is_file(GLPI_DOC_DIR . "/" . $input['current_filepath'])
            && (countElementsInTable(
                    'glpi_documents',
                    ['sha1sum' => sha1_file(GLPI_DOC_DIR . "/" .
                        $input['current_filepath'])
                    ]
                ) <= 1)
        ) {
            if (unlink(GLPI_DOC_DIR . "/" . $input['current_filepath'])) {
                Session::addMessageAfterRedirect(sprintf(
                    __('Successful deletion of the file %s'),
                    $input['current_filename']
                ));
            } else {
                // TRANS: %1$s is the curent filename, %2$s is its directory
                trigger_error(
                    sprintf(
                        'Failed to delete the file %1$s (%2$s)',
                        $input['current_filename'],
                        GLPI_DOC_DIR . "/" . $input['current_filepath']
                    ),
                    E_USER_WARNING
                );
                Session::addMessageAfterRedirect(
                    sprintf(
                        __('Failed to delete the file %1$s'),
                        $input['current_filename']
                    ),
                    false,
                    ERROR
                );
            }
        }

        // Local file : try to detect mime type
        $input['mime'] = Toolbox::getMime($fullpath);

        if (
            is_writable(GLPI_UPLOAD_DIR)
            && is_writable($fullpath)
        ) { // Move if allowed
            if (self::renameForce($fullpath, GLPI_DOC_DIR . "/" . $new_path)) {

                $wordPath = GLPI_DOC_DIR . "/" . $new_path;
                $pdfPath = dirname($wordPath) . "/" . pathinfo($wordPath, PATHINFO_FILENAME) . '.pdf';

                if (self::convertWordToPDF($wordPath, $pdfPath)) {
                    Session::addMessageAfterRedirect(__('Document move and conversion to PDF succeeded.'));
                } else {
                    Session::addMessageAfterRedirect(__('Document move succeeded, but PDF conversion failed.'), false, ERROR);
                }

//                Session::addMessageAfterRedirect(__('Document move succeeded.'));
            } else {
                Session::addMessageAfterRedirect(__('File move failed.'), false, ERROR);
                return false;
            }
        } else { // Copy (will overwrite dest file is present)
            if (copy($fullpath, GLPI_DOC_DIR . "/" . $new_path)) {

                $wordPath = GLPI_DOC_DIR . "/" . $new_path;
                $pdfPath = dirname($wordPath) . "/" . pathinfo($wordPath, PATHINFO_FILENAME) . '.pdf';

                if (static::convertWordToPDF($wordPath, $pdfPath)) {
                    Session::addMessageAfterRedirect(__('Document move and conversion to PDF succeeded.'));
                } else {
                    Session::addMessageAfterRedirect(__('Document move succeeded, but PDF conversion failed.'), false, ERROR);
                }

//                Session::addMessageAfterRedirect(__('Document copy succeeded.'));
            } else {
                Session::addMessageAfterRedirect(__('File move failed'), false, ERROR);
                return false;
            }
        }

        // For display
        $input['filename'] = addslashes($filename);
        // Storage path
        $input['filepath'] = $new_path;
        // Checksum
        $input['sha1sum']  = $sha1sum;
        return true;
    }


//    dlteams
    static function convertWordToPDF($inputPath, $outputPath) {
        $command = "libreoffice --headless --convert-to pdf --outdir " . escapeshellarg(dirname($outputPath)) . " " . escapeshellarg($inputPath);
        exec($command, $output, $returnCode);

        return $returnCode === 0 && file_exists($outputPath);
    }

    /**
     * Move a document (files in GLPI_DOC_DIR."/_tmp" dir)
     *
     * @param array  $input     array of datas used in adding process (need current_filepath)
     * @param string $filename  filename to move
     *
     * @return boolean for success / $input array is updated
     **/
    public static function moveDocument(array &$input, $filename)
    {
        if (str_contains($filename, '/') || str_contains($filename, '\\')) {
            // Filename is not supposed to contains directory separators.
            trigger_error(sprintf('Moving file `%s` is forbidden for security reasons.', $filename), E_USER_WARNING);
            return false;
        }

        $prefix = '';
        if (isset($input['_prefix_filename'])) {
            $prefix = array_shift($input['_prefix_filename']);
        }

        $fullpath = GLPI_TMP_DIR . "/" . $filename;
        $filename = str_replace($prefix, '', $filename);
        if (!is_dir(GLPI_TMP_DIR)) {
            Session::addMessageAfterRedirect(__("Temporary directory doesn't exist"), false, ERROR);
            return false;
        }

        if (!is_file($fullpath)) {
            trigger_error(
                sprintf('File %s not found.', $fullpath),
                E_USER_WARNING
            );
            Session::addMessageAfterRedirect(
                sprintf(__('File %s not found.'), $filename),
                false,
                ERROR
            );
            return false;
        }
        $sha1sum  = sha1_file($fullpath);
        $dir      = self::isValidDoc($filename);
        $new_path = self::getUploadFileValidLocationName($dir, $sha1sum);

        if (!$sha1sum || !$dir || !$new_path) {
            @unlink($fullpath);
            return false;
        }

        // Delete old file (if not used by another doc)
        if (
            isset($input['current_filepath'])
            && !empty($input['current_filepath'])
            && is_file(GLPI_DOC_DIR . "/" . $input['current_filepath'])
            && (countElementsInTable(
                    'glpi_documents',
                    ['sha1sum' => sha1_file(GLPI_DOC_DIR . "/" .
                        $input['current_filepath'])
                    ]
                ) <= 1)
        ) {
            if (unlink(GLPI_DOC_DIR . "/" . $input['current_filepath'])) {
                Session::addMessageAfterRedirect(sprintf(
                    __('Successful deletion of the file %s'),
                    $input['current_filename']
                ));
            } else {
                // TRANS: %1$s is the curent filename, %2$s is its directory
                trigger_error(
                    sprintf(
                        'Failed to delete the file %1$s (%2$s)',
                        $input['current_filename'],
                        GLPI_DOC_DIR . "/" . $input['current_filepath']
                    ),
                    E_USER_WARNING
                );
                Session::addMessageAfterRedirect(
                    sprintf(
                        __('Failed to delete the file %1$s'),
                        $input['current_filename']
                    ),
                    false,
                    ERROR
                );
            }
        }

        // Local file : try to detect mime type
        $input['mime'] = Toolbox::getMime($fullpath);


        // Copy (will overwrite dest file if present)
        if (copy($fullpath, GLPI_DOC_DIR . "/" . $new_path)) {


            $inputFile = GLPI_DOC_DIR . "/" . $new_path;
            $outputDir = dirname($inputFile);

            $command = 'libreoffice --headless --convert-to pdf --outdir '
                . escapeshellarg($outputDir) . ' '
                . escapeshellarg($inputFile) . ' 2>&1';


            $output = [];
            $returnVar = 0;
            exec($command, $output, $returnVar);



            Session::addMessageAfterRedirect(__('Document copy succeeded.'));
        } else {
            Session::addMessageAfterRedirect(__('File move failed'), false, ERROR);
            @unlink($fullpath);
            return false;
        }

        // For display
        $input['filename'] = addslashes($filename);
        // Storage path
        $input['filepath'] = $new_path;
        // Checksum
        $input['sha1sum']  = $sha1sum;
        return true;
    }


    /**
     * Upload a new file
     *
     * @param &$input    array of datas need for add/update (will be completed)
     * @param $FILEDESC        FILE descriptor
     *
     * @return boolean
     **/
    public static function uploadDocument(array &$input, $FILEDESC)
    {

        if (
            !count($FILEDESC)
            || empty($FILEDESC['name'])
            || !is_file($FILEDESC['tmp_name'])
        ) {
            switch ($FILEDESC['error']) {
                case 1:
                case 2:
                    Session::addMessageAfterRedirect(__('File too large to be added.'), false, ERROR);
                    break;

                case 4:
                    // Session::addMessageAfterRedirect(__('No file specified.'),false,ERROR);
                    break;
            }

            return false;
        }

        $sha1sum = sha1_file($FILEDESC['tmp_name']);
        $dir     = self::isValidDoc($FILEDESC['name']);
        $path    = self::getUploadFileValidLocationName($dir, $sha1sum);

        if (!$sha1sum || !$dir || !$path) {
            return false;
        }

        // Delete old file (if not used by another doc)
        if (
            isset($input['current_filepath'])
            && !empty($input['current_filepath'])
            && (countElementsInTable(
                    'glpi_documents',
                    ['sha1sum' => sha1_file(GLPI_DOC_DIR . "/" .
                        $input['current_filepath'])
                    ]
                ) <= 1)
        ) {
            if (unlink(GLPI_DOC_DIR . "/" . $input['current_filepath'])) {
                Session::addMessageAfterRedirect(sprintf(
                    __('Successful deletion of the file %s'),
                    $input['current_filename']
                ));
            } else {
                // TRANS: %1$s is the curent filename, %2$s is its directory
                trigger_error(
                    sprintf(
                        'Failed to delete the file %1$s (%2$s)',
                        $input['current_filename'],
                        GLPI_DOC_DIR . "/" . $input['current_filepath']
                    ),
                    E_USER_WARNING
                );
                Session::addMessageAfterRedirect(
                    sprintf(
                        __('Failed to delete the file %1$s'),
                        $input['current_filename']
                    ),
                    false,
                    ERROR
                );
            }
        }

        // Mime type from client
        if (isset($FILEDESC['type']) && !empty($FILEDESC['type'])) {
            $input['mime'] = $FILEDESC['type'];
        }

        // Move uploaded file
        if (self::renameForce($FILEDESC['tmp_name'], GLPI_DOC_DIR . "/" . $path)) {
            Session::addMessageAfterRedirect(__('The file is valid. Upload is successful.'));
            // For display
            $input['filename'] = addslashes($FILEDESC['name']);
            // Storage path
            $input['filepath'] = $path;
            // Checksum
            $input['sha1sum']  = $sha1sum;
            return true;
        }
        Session::addMessageAfterRedirect(
            __('Potential upload attack or file too large. Moving temporary file failed.'),
            false,
            ERROR
        );
        return false;
    }


    /**
     * Find a valid path for the new file
     *
     * @param string $dir      dir to search a free path for the file
     * @param string $sha1sum  SHA1 of the file
     *
     * @return string
     **/
    public static function getUploadFileValidLocationName($dir, $sha1sum)
    {
        if (empty($dir)) {
            $message = __('Unauthorized file type');

            if (Session::haveRight('dropdown', READ)) {
                $dt       = new DocumentType();
                $message .= " <a target='_blank' href='" . $dt->getSearchURL() . "' class='pointer'>
                         <i class='fa fa-info'</i><span class='sr-only'>" . __('Manage document types')  . "</span></a>";
            }
            Session::addMessageAfterRedirect($message, false, ERROR);
            return '';
        }

        if (!is_dir(GLPI_DOC_DIR)) {
            trigger_error(
                sprintf(
                    "The directory %s doesn't exist.",
                    GLPI_DOC_DIR
                ),
                E_USER_WARNING
            );
            Session::addMessageAfterRedirect(
                sprintf(
                    __("Documents directory doesn't exist.")
                ),
                false,
                ERROR
            );
            return '';
        }
        $subdir = $dir . '/' . substr($sha1sum, 0, 2);

        if (
            !is_dir(GLPI_DOC_DIR . "/" . $subdir)
            && @mkdir(GLPI_DOC_DIR . "/" . $subdir, 0777, true)
        ) {
            Session::addMessageAfterRedirect(sprintf(
                __('Create the directory %s'),
                $subdir
            ));
        }

        if (!is_dir(GLPI_DOC_DIR . "/" . $subdir)) {
            trigger_error(
                sprintf(
                    'Failed to create the directory %s.',
                    GLPI_DOC_DIR . "/" . $subdir
                ),
                E_USER_WARNING
            );
            Session::addMessageAfterRedirect(
                sprintf(
                    __('Failed to create the directory %s. Verify that you have the correct permission'),
                    $subdir
                ),
                false,
                ERROR
            );
            return '';
        }
        return $subdir . '/' . substr($sha1sum, 2) . '.' . $dir;
    }


    /**
     * Show dropdown of uploaded files
     *
     * @param $myname dropdown name
     **/
    public static function showUploadedFilesDropdown($myname)
    {
        if (is_dir(GLPI_UPLOAD_DIR)) {
            $uploaded_files = [];
            if ($handle = opendir(GLPI_UPLOAD_DIR)) {
                while (false !== ($file = readdir($handle))) {
                    if (!in_array($file, ['.', '..', '.gitkeep', 'remove.txt'])) {
                        $dir = self::isValidDoc($file);
                        if (!empty($dir)) {
                            $uploaded_files[$file] = $file;
                        }
                    }
                }
                closedir($handle);
            }

            if (count($uploaded_files)) {
                Dropdown::showFromArray($myname, $uploaded_files, ['display_emptychoice' => true]);
            } else {
                echo __('No file available');
            }
        } else {
            echo __("Upload directory doesn't exist");
        }
    }


    /**
     * Is this file a valid file ? check based on file extension
     *
     * @param string $filename filename to clean
     **/
    public static function isValidDoc($filename)
    {
        /** @var \DBmysql $DB */
        global $DB;

        $splitter = explode(".", $filename);
        $ext      = end($splitter);

        $iterator = $DB->request([
            'FROM'   => 'glpi_documenttypes',
            'WHERE'  => [
                'ext'             => ['LIKE', $ext],
                'is_uploadable'   => 1
            ]
        ]);

        if (count($iterator)) {
            return Toolbox::strtoupper($ext);
        }

        // Not found try with regex one
        $iterator = $DB->request([
            'FROM'   => 'glpi_documenttypes',
            'WHERE'  => [
                'ext'             => ['LIKE', '/%/'],
                'is_uploadable'   => 1
            ]
        ]);

        foreach ($iterator as $data) {
            if (preg_match(Sanitizer::unsanitize($data['ext']) . "i", $ext, $results) > 0) {
                return Toolbox::strtoupper($ext);
            }
        }

        return "";
    }

    /**
     * Make a select box for link document
     *
     * Parameters which could be used in options array :
     *    - name : string / name of the select (default is documents_id)
     *    - entity : integer or array / restrict to a defined entity or array of entities
     *                   (default -1 : no restriction)
     *    - used : array / Already used items ID: not to display in dropdown (default empty)
     *    - hide_if_no_elements  : boolean / hide dropdown if there is no elements (default false)
     *
     * @param $options array of possible options
     *
     * @return integer|string
     *    integer if option display=true (random part of elements id)
     *    string if option display=false (HTML code)
     **/
    public static function dropdown($options = [])
    {
        /**
         * @var array $CFG_GLPI
         * @var \DBmysql $DB
         */
        global $CFG_GLPI, $DB;

        $p['name']    = 'documents_id';
        $p['entity']  = '';
        $p['used']    = [];
        $p['display'] = true;
        $p['hide_if_no_elements'] = false;
        $p['readonly'] = false;

        if (is_array($options) && count($options)) {
            foreach ($options as $key => $val) {
                $p[$key] = $val;
            }
        }

        if (isset($p['value']) && ($p['value'] > 0)) {
            $document = new Document();
            $document->getFromDB($p['value']);
            $p['rubdoc'] = $document->fields['documentcategories_id'];
        }

        $subwhere = [
                'glpi_documents.is_deleted'   => 0,
            ] + getEntitiesRestrictCriteria('glpi_documents', '', $p['entity'], true);

        if (count($p['used'])) {
            $subwhere['NOT'] = ['id' => array_merge([0], $p['used'])];
        }

        $criteria = [
            'FROM'   => 'glpi_documentcategories',
            'WHERE'  => [
                'id' => new QuerySubQuery([
                    'SELECT'          => 'documentcategories_id',
                    'DISTINCT'        => true,
                    'FROM'            => 'glpi_documents',
                    'WHERE'           => $subwhere
                ])
            ],
            'ORDER'  => 'name'
        ];
        $iterator = $DB->request($criteria);

        if ($p['hide_if_no_elements'] && $iterator->count() === 0) {
            return;
        }

        $values = [];
        foreach ($iterator as $data) {
            $values[$data['id']] = $data['name'];
        }
        $rand = mt_rand();
        $readonly = $p['readonly'];
        $out = '';
        $width = '30%';
        if ($readonly) {
            $width = '100%';
            $out .= '<div class="row">';
            $out .= '<div class="col-xxl-5 p-0">';
        }
        $out  .= Dropdown::showFromArray('_rubdoc', $values, [
            'width'               => $width,
            'rand'                => $rand,
            'display'             => false,
            'display_emptychoice' => true,
            'value'               => $p['rubdoc'] ?? 0,
            'readonly'            => $readonly
        ]);
        $field_id = Html::cleanId("dropdown__rubdoc$rand");

        $params   = ['rubdoc' => '__VALUE__',
            'entity' => $p['entity'],
            'rand'   => $rand,
            'myname' => $p['name'],
            'used'   => $p['used']
        ];

        if ($readonly) {
            $out .= '</div>';
            $out .= '<div class="col-xxl-7 p-0">';
        }
        $out .= Ajax::updateItemOnSelectEvent(
            $field_id,
            "show_" . $p['name'] . $rand,
            $CFG_GLPI["root_doc"] . "/ajax/dropdownRubDocument.php",
            $params,
            false
        );
        $out .= "<span id='show_" . $p['name'] . "$rand'>";
        $out .= "</span>\n";

        $params['rubdoc'] = $p['rubdoc'] ?? 0;
        $params['value'] = $p['value'] ?? 0;
        if ($readonly) {
            $document = new Document();
            $doclist = $document->find([]);
            foreach ($doclist as $doc) {
                $docvalue[$doc['id']] = $doc['name'];
            }

            $out .= Dropdown::showFromArray('document', $docvalue ?? [], [
                'width'               => $width,
                'rand'                => $rand,
                'display'             => false,
                'display_emptychoice' => true,
                'value'               => $p['value'] ?? 0,
                'readonly'            => $readonly
            ]);
            $out .= '</div>';
            $out .= '</div>';
        } else {
            $out .= Ajax::updateItem(
                "show_" . $p['name'] . $rand,
                $CFG_GLPI["root_doc"] . "/ajax/dropdownRubDocument.php",
                $params,
                false
            );
        }
        if ($p['display']) {
            echo $out;
            return $rand;
        }
        return $out;
    }


    public static function getMassiveActionsForItemtype(
        array &$actions,
        $itemtype,
        $is_deleted = false,
        ?CommonDBTM $checkitem = null
    ) {
        $action_prefix = 'Document_Item' . MassiveAction::CLASS_ACTION_SEPARATOR;

        if (self::canApplyOn($itemtype)) {
            if (Document::canView()) {
                $actions[$action_prefix . 'add']    = "<i class='fa-fw " . self::getIcon() . "'></i>" .
                    _x('button', 'Add a document');
                $actions[$action_prefix . 'remove'] = _x('button', 'Remove a document');
            }
        }

        if ((is_a($itemtype, __CLASS__, true)) && (static::canUpdate())) {
            $actions[$action_prefix . 'add_item']    = _x('button', 'Add an item');
            $actions[$action_prefix . 'remove_item'] = _x('button', 'Remove an item');
        }
    }


    /**
     * @since 0.85
     *
     * @param $string
     *
     * @return string
     **/
    public static function getImageTag($string)
    {
        return self::$tag_prefix . $string . self::$tag_prefix;
    }

    /**
     * Is file an image
     *
     * @since 9.2.1
     *
     * @param string $file File name
     *
     * @return boolean
     */
    public static function isImage($file)
    {
        if (!file_exists($file)) {
            return false;
        }
        if (extension_loaded('exif')) {
            if (filesize($file) < 12) {
                return false;
            }
            $etype = exif_imagetype($file);
            return in_array($etype, [IMAGETYPE_JPEG, IMAGETYPE_GIF, IMAGETYPE_PNG, IMAGETYPE_BMP, IMAGETYPE_WEBP]);
        } else {
            trigger_error(
                'For security reasons, you should consider using exif PHP extension to properly check images.',
                E_USER_WARNING
            );
            $fileinfo = finfo_open(FILEINFO_MIME_TYPE);
            return in_array(
                finfo_file($fileinfo, $file),
                ['image/jpeg', 'image/png','image/gif', 'image/bmp', 'image/webp']
            );
        }
    }

    /**
     * Get image path for a specified context.
     * Will call image resize if needed.
     *
     * @since 9.2.1
     *
     * @param string  $path    Original path
     * @param string  $context Context
     * @param integer $mwidth  Maximal width
     * @param integer $mheight Maximal height
     *
     * @return string Image path on disk
     */
    public static function getImage($path, $context, $mwidth = null, $mheight = null)
    {
        if ($mwidth === null || $mheight === null) {
            switch ($context) {
                case 'mail':
                    $mwidth  = $mwidth ?? 400;
                    $mheight = $mheight ?? 300;
                    break;
                case 'timeline':
                    $mwidth  = $mwidth ?? 100;
                    $mheight = $mheight ?? 100;
                    break;
                default:
                    throw new \RuntimeException("Unknown context $context!");
            }
        }

        //let's see if original image needs resize
        $img_infos  = getimagesize($path);
        if (!($img_infos[0] > $mwidth) && !($img_infos[1] > $mheight)) {
            //no resize needed
            return $path;
        }

        $infos = pathinfo($path);
        // output images with possible transparency to png, other to jpg
        $extension = in_array(strtolower($infos['extension']), ['png', 'gif']) ? 'png' : 'jpg';
        $context_path = sprintf(
            '%1$s_%2$s-%3$s.%4$s',
            $infos['dirname'] . '/' . $infos['filename'],
            $mwidth,
            $mheight,
            $extension
        );

        //let's check if file already exists
        if (file_exists($context_path)) {
            return $context_path;
        }

        //do resize
        $result = Toolbox::resizePicture(
            $path,
            $context_path,
            $mwidth,
            $mheight,
            0,
            0,
            0,
            0,
            ($mwidth > $mheight ? $mwidth : $mheight)
        );
        return ($result ? $context_path : $path);
    }

    /**
     * Give cron information
     *
     * @param string $name task's name
     *
     * @return array of information
     **/
    public static function cronInfo($name)
    {

        switch ($name) {
            case 'cleanorphans':
                return ['description' => __('Clean orphaned documents: deletes all documents that are not associated with any items.')];
        }
        return [];
    }

    /**
     * Cron for clean orphan documents (without Document_Item)
     *
     * @param CronTask $task CronTask object
     *
     * @return integer (0 : nothing done - 1 : done)
     **/
    public static function cronCleanOrphans(CronTask $task)
    {
        /** @var \DBmysql $DB */
        global $DB;

        $dtable = static::getTable();
        $ditable = Document_Item::getTable();
        //documents that are not present in Document_Item are oprhan
        $iterator = $DB->request([
            'SELECT'    => ["$dtable.id"],
            'FROM'      => $dtable,
            'LEFT JOIN' => [
                $ditable => [
                    'ON'  => [
                        $dtable  => 'id',
                        $ditable => 'documents_id'
                    ]
                ]
            ],
            'WHERE'     => [
                "$ditable.documents_id" => null
            ]
        ]);

        $nb = 0;
        if (count($iterator)) {
            foreach ($iterator as $row) {
                $doc = new Document();
                $doc->delete(['id' => $row['id']], true);
                ++$nb;
            }
        }

        if ($nb) {
            $task->addVolume($nb);
            $task->log("Documents : $nb");
        }

        return ($nb > 0 ? 1 : 0);
    }


    public static function getIcon()
    {
        return "ti ti-files";
    }


    /**
     * find and load a document which is a duplicate of a file, with respect of blacklisting
     *
     * @param integer $entities_id  entity of the document
     * @param string  $filename     filename of the searched file
     *
     * @return boolean
     */
    public function getDuplicateOf(int $entities_id, string $filename): bool
    {
        if (!$this->getFromDBbyContent($entities_id, $filename)) {
            return false;
        }

        if ($this->fields['is_blacklisted']) {
            return false;
        }

        return true;
    }


    /**
     * It checks if a file exists and is readable
     *
     * @param string $filename The name of the file to check.
     *
     * @return boolean
     */
    public function checkAvailability(string $filename): bool
    {
        $file = GLPI_DOC_DIR . '/' . $filename;
        if (!file_exists($file)) {
            return false;
        }

        if (!is_readable($file)) {
            return false;
        }

        return true;
    }

//    dlteams
    /**
     * Get list of document categories in fancytree format.
     *
     * @param class-string<CommonDBTM> $itemtype
     * @param array $params
     *
     * @return array
     */
    public static function getTreeCategoryList(string $itemtype, array $params): array
    {
        /** @var \DBmysql $DB */
        global $DB;

        /** @var class-string<CommonDBTM> $cat_itemtype */
        $cat_itemtype = static::getCategoryItemType($itemtype);
        $cat_item     = new $cat_itemtype();

        $params['export_all'] = true;
        $data = Search::prepareDatasForSearch($itemtype, $params);
        Search::constructSQL($data);
        // This query is used to get the IDs of all results matching the search criteria
        $sql = $data['sql']['search'];
        // We can remove all the SELECT fields and replace it with just the ID field
        $raw_select = $data['sql']['raw']['SELECT'];
        $replacement_select = 'SELECT DISTINCT ' . $itemtype::getTableField('id');
        $sql = preg_replace('/^' . preg_quote($raw_select, '/') . '/', $replacement_select, $sql, 1);
        // Remove GROUP BY and ORDER BY clauses
        $sql = str_replace([$data['sql']['raw']['GROUPBY'], $data['sql']['raw']['ORDER']], '', $sql);

        $id_criteria = new QueryExpression($itemtype::getTableField('id') . ' IN ( SELECT * FROM (' . $sql . ') AS id_criteria )');

        $cat_table = $cat_itemtype::getTable();
        $cat_fk    = $cat_itemtype::getForeignKeyField();

        $items_subquery = new QuerySubQuery(
            [
                'SELECT' => ['COUNT DISTINCT' => $itemtype::getTableField('id') . ' AS cpt'],
                'FROM'   => $itemtype::getTable(),
                'WHERE'  => [
                    $itemtype::getTableField($cat_fk) => new QueryExpression($DB::quoteName($cat_itemtype::getTableField('id'))),
                    $id_criteria,
                ]
            ],
            'items_count'
        );

        /*        highlight_string("<?php\n\$data =\n" . var_export($items_subquery->getQuery(), true) . ";\n?>");*/
//        die();

        $select[] = $cat_itemtype::getTableField('id');
        $select[] = $cat_itemtype::getTableField('name');
        if ($cat_item instanceof CommonTreeDropdown) {
            $select[] = $cat_itemtype::getTableField($cat_fk);
        }
        $select[] = $items_subquery;

        if ($cat_item instanceof CommonTreeDropdown) {
            $order[] = $cat_itemtype::getTableField('level') . ' DESC';
            $order[] = $cat_itemtype::getTableField('name');
        } else {
            $order[] = $cat_itemtype::getTableField('name') . ' DESC';
        }

        $cat_iterator = $DB->request([
            'SELECT' => $select,
            'FROM' => $cat_table,
            'ORDER' => $order,
            'WHERE' => [
                "glpi_documentcategories.entities_id" => $_SESSION['glpiactive_entity']
            ]
        ]);
        /*        highlight_string("<?php\n\$data =\n" . var_export($cat_table, true) . ";\n?>");*/
//        die();

        $inst = new $cat_itemtype();
        $categories = [];
        foreach ($cat_iterator as $category) {
            if (DropdownTranslation::canBeTranslated($inst)) {
                $tname = DropdownTranslation::getTranslatedValue(
                    $category['id'],
                    $inst->getType()
                );
                if (!empty($tname)) {
                    $category['name'] = $tname;
                }
            }
            $categories[] = $category;
        }

        // Without category
        $no_cat_count = $DB->request(
            [
                'SELECT' => ['COUNT DISTINCT' => $itemtype::getTableField('id') . ' as cpt'],
                'FROM'   => $itemtype::getTable(),
                'WHERE'  => [
                    $itemtype::getTableField($cat_fk) => 0,
                    $id_criteria,
                ]
            ]
        )->current();
        if ($no_cat_count['cpt'] > 0) {
            $categories[] = [
                'id'          => -1,
                'name'        => __s('Without Category'),
                'items_count' => $no_cat_count['cpt'],
                $cat_fk       => 0,
            ];
        }

        // construct flat data
        $nodes   = [];
        foreach ($categories as $category) {
            $cat_id = $category['id'];
            $node = [
                'key'    => $cat_id,
                'title'  => $category['name'],
                'parent' => $category[$cat_fk] ?? 0,
                'a_attr' => [
                    'data-id' => $cat_id
                ],
            ];

            if ($category['items_count'] > 0) {
                $node['title'] .= ' <span class="badge bg-azure-lt" title="' . __s('This category contains ') . $itemtype::getTypeName() . '">'
                    . $category['items_count']
                    . '</span>';
            }

            $nodes[] = $node;
        }

        // recursive construct tree data
        $buildtree = function (array &$elements, $parent = 0) use (&$buildtree) {
            $branch = [];

            foreach ($elements as $index => $element) {
                if ($element['parent'] === $parent) {
                    $children = $buildtree($elements, $element['key']);
                    if (count($children) > 0) {
                        $element['children'] = $children;
                    }
                    $branch[] = $element;
                    unset($elements[$index]);
                }
            }
            return $branch;
        };

        $newtree = $buildtree($nodes);

        return $newtree;
    }


    public static function showMassiveActionsSubForm(MassiveAction $ma)
    {
        switch ($ma->getAction()) {
            case 'copyTo':
                Entity::dropdown([
                    'name' => 'entities_id',
                ]);
                echo '<br>';
                echo "<table class='tab_cadre_fixe' style='width: auto'>\n";
                echo "<tr>";
                echo "<td class='right'><input class='form-check-input ms-0 me-1' type='checkbox' name='copy_filepath' value='1'></td>";
                echo "<td class='left'>" . __('Copier aussi le fichier lié') . "</td>";
                echo "</tr>";

                echo "<tr>";
                echo "<td class='right'><input class='form-check-input ms-0 me-1' type='checkbox' name='copy_deliverable' value='1'></td>";
                echo "<td class='left'>" . __('Copier aussi le fichier HTML') . "</td>";
                echo "</tr>";

                echo "<tr>";
                echo "<td class='right'><input class='form-check-input ms-0 me-1' type='checkbox' name='copy_section' value='1'></td>";
                echo "<td class='left'>" . __('Créer la rubrique si inexistante') . "</td>";
                echo "</tr>";
                echo "</table>";

                echo '<br /><br />' . Html::submit(_x('button', 'Post'), ['name' => 'massiveaction']);
                return true;
        }
        return parent::showMassiveActionsSubForm($ma);
    }

    static function processMassiveActionsForOneItemtype(MassiveAction $ma, CommonDBTM $item, array $ids)
    {
        switch ($ma->getAction()) {
            case 'copyTo':
                if ($item->getType() == Document::getType()) {
                    foreach ($ids as $id) {
                        if ($item->getFromDB($id)) {
                            if ($item->copy1($ma->POST['entities_id'], $id, $item)) {

                                Session::addMessageAfterRedirect(sprintf(__('%s copied: %s', 'dlteams'), $item->getTypeName(), $item->getName()));
                                $ma->itemDone($item->getType(), $id, MassiveAction::ACTION_OK);
                            }
                        } else {
                            // Example of ko count
                            $ma->itemDone($item->getType(), $id, MassiveAction::ACTION_KO);
                        }
                    }
                }
                return;
        }
        parent::processMassiveActionsForOneItemtype($ma, $item, $ids);
    }


    public function copy1($entity, $id, $item){
        /*        highlight_string("<?php\n\$data =\n" . var_export($_POST, true) . ";\n?>");*/
//        die();
        global $DB;
        $dbu = new DbUtils();
        $name=$item->fields['name'];



        $nb=$dbu->countElementsInTable(static::getTable(), ['name' => addslashes($name??""), 'entities_id' => $entity]);

        if(empty($name) || $nb<=0){
            $data = $item->fields;
            unset($data["id"]);
            unset($data["date_creation"]);
            unset($data["date_mod"]);
            if(!isset($_POST["copy_filepath"]) || !$_POST["copy_filepath"]){
                unset($data["filename"]);
                unset($data["file_path"]);
            }

//            if(!$_POST["copy_deliverable"]){
//                unset($data["deliverables_id"]);
////                unset($data["file_path"]);
//            }


            $data["entities_id"] = $entity;

//            $DB->request("INSERT INTO ".static::getTable()." (entities_id, is_recursive, date_mod, date_creation, name, content, comment, plugin_dlteams_rightmeasurecategories_id) SELECT '$entity', is_recursive, date_mod, date_creation, name, content, comment, 0 FROM ".static::getTable()." WHERE id='$id'");
            $document = new Document();
            $documents_id = $document->add($data);

            if(isset($_POST["copy_deliverable"]) && $_POST["copy_deliverable"] && $data["deliverables_id"]){
                $deliverable = new PluginDlteamsDeliverable();
                $deliverable->getFromDB($data["deliverables_id"]);
                $data2 = $deliverable->fields;

                $data2["entities_id"] = $entity;
                $data2["documents_id"] = $documents_id;
                unset($data2["id"]);
                unset($data2["date_creation"]);
                unset($data2["date_mod"]);

                $deliverables_id = $deliverable->add([
                    ...$data2,
                    "name" => addslashes($data["name"] ?? ""),
                    "content" => addslashes($data["content"] ?? ""),
                    "comment" => addslashes($data["comment"] ?? ""),
                    "object_approval" => addslashes($data["object_approval"] ?? ""),
                    "object_notification" => addslashes($data["object_notification"] ?? ""),
                    "document_content" => addslashes($data["document_content"] ?? ""),
                    "document_title" => addslashes($data["document_title"] ?? ""),
                    "document_comment" => addslashes($data["document_comment"] ?? ""),
                    "text_approval" => addslashes($data["text_approval"] ?? ""),
                    "text_notification" => addslashes($data["text_notification"] ?? ""),
                ]);
                $document->update([
                    "deliverables_id" => $deliverables_id,
                    "id" => $documents_id
                ]);
                Session::addMessageAfterRedirect("Fichier HTML lié copié avec succès");
                return true;
            }

            return false;
        }
        else{
            Session::addMessageAfterRedirect("Un document existe portant le meme nom", 0, ERROR);
            return false;
        }
    }
}
