<?php

/**
 * ---------------------------------------------------------------------
 *
 * GLPI - Gestionnaire Libre de Parc Informatique
 *
 * http://glpi-project.org
 *
 * @copyright 2015-2024 Teclib' and contributors.
 * @copyright 2003-2014 by the INDEPNET Development Team.
 * @licence   https://www.gnu.org/licenses/gpl-3.0.html
 *
 * ---------------------------------------------------------------------
 *
 * LICENSE
 *
 * This file is part of GLPI.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * ---------------------------------------------------------------------
 */

/**
 * TicketValidation class
 */
class TicketValidation extends CommonITILValidation
{
   // From CommonDBChild
    public static $itemtype           = 'Ticket';
    public static $items_id           = 'tickets_id';

    public static $rightname                 = 'ticketvalidation';

    const CREATEREQUEST               = 1024;
    const CREATEINCIDENT              = 2048;
    const VALIDATEREQUEST             = 4096;
    const VALIDATEINCIDENT            = 8192;



    public static function getCreateRights()
    {
        return [static::CREATEREQUEST, static::CREATEINCIDENT];
    }


    public static function getValidateRights()
    {
        return [static::VALIDATEREQUEST, static::VALIDATEINCIDENT];
    }


    /**
     * @since 0.85
     **/
    public function canCreateItem()
    {

        if ($this->canChildItem('canViewItem', 'canView')) {
            $ticket = new Ticket();
            if ($ticket->getFromDB($this->fields['tickets_id'])) {
                // No validation for closed tickets
                if (in_array($ticket->fields['status'], $ticket->getClosedStatusArray())) {
                    return false;
                }

                if ($ticket->fields['type'] == Ticket::INCIDENT_TYPE) {
                    return Session::haveRight(self::$rightname, self::CREATEINCIDENT);
                }
                if ($ticket->fields['type'] == Ticket::DEMAND_TYPE) {
                    return Session::haveRight(self::$rightname, self::CREATEREQUEST);
                }
            }
        }

        return parent::canCreateItem();
    }

    /**
     * @since 0.85
     *
     * @see commonDBTM::getRights()
     **/
    public function getRights($interface = 'central')
    {

        $values = parent::getRights();
        unset($values[UPDATE], $values[CREATE], $values[READ]);

        $values[self::CREATEREQUEST]
                              = ['short' => __('Create for request'),
                                  'long'  => __('Create a validation request for a request')
                              ];
        $values[self::CREATEINCIDENT]
                              = ['short' => __('Create for incident'),
                                  'long'  => __('Create a validation request for an incident')
                              ];
        $values[self::VALIDATEREQUEST]
                              = __('Validate a request');
        $values[self::VALIDATEINCIDENT]
                              = __('Validate an incident');

        if ($interface == 'helpdesk') {
            unset($values[PURGE]);
        }

        return $values;
    }


//    fork dlteams

    public function getTabNameForItem(CommonGLPI $item, $withtemplate = 0)
    {
        /** @var CommonDBTM $item */
        $hidetab = false;
        // Hide if no rights on validations
        if (!static::canView()) {
            $hidetab = true;
        }
        // No right to create and no validation for current object
        if (
            !$hidetab
            && !Session::haveRightsOr(static::$rightname, static::getCreateRights())
            && !static::canValidate($item->getID())
        ) {
            $hidetab = true;
        }

        if (!$hidetab) {
            $nb = 0;
            if ($_SESSION['glpishow_count_on_tabs']) {
                $restrict = [static::$items_id => $item->getID()];
                // No rights for create only count asign ones
                if (!Session::haveRightsOr(static::$rightname, static::getCreateRights())) {
                    $restrict['users_id_validate'] = Session::getLoginUserID();
                }


                global $DB;
                $iterator = $DB->Request([
                    'FROM'   => $this->getTable(),
                    'WHERE'  => [static::$items_id => $item->getField('id')],
                    'ORDER'  => 'submission_date DESC'
                ]);

                $nb = count($iterator);




                $tickettasks_forticket_query = [
                    "FROM" => TicketTask::getTable(),
                    "WHERE" => [
                        "tickets_id" => $item->getField('id')
                    ]
                ];
//        var_dump("zzz");
//        die();
                $tickettasks_forticket_iterator = $DB->request($tickettasks_forticket_query);
                $tickettasks_idx = array_column(iterator_to_array($tickettasks_forticket_iterator), 'id');
                if(count($tickettasks_idx)>0) {
                    $iterator = $DB->Request([
                        'FROM' => PluginDlteamsTicketTask_Validation::getTable(),
                        'WHERE' => [
                            "tickettasks_id" => $tickettasks_idx
                        ],
                        'ORDER' => 'submission_date DESC'
                    ]);
                    $nb+=count($iterator);
                }

            }
            return self::createTabEntry(self::getTypeName(Session::getPluralNumber()), $nb);
        }
        return '';
    }


    public static function displayTabContentForItem(CommonGLPI $item, $tabnum = 1, $withtemplate = 0)
    {

        $validation = new static();
        $validation->showSummary($item);
        return true;
    }


    /**
     * Print the validation list into item
     *
     * @param CommonDBTM $item
     **/
    public function showSummary(CommonDBTM $item)
    {
        /**
         * @var array $CFG_GLPI
         * @var \DBmysql $DB
         */
        global $CFG_GLPI, $DB;

        if (
        !Session::haveRightsOr(
            static::$rightname,
            array_merge(
                static::getCreateRights(),
                static::getValidateRights(),
                static::getPurgeRights()
            )
        )
        ) {
            return false;
        }

        $tID    = $item->fields['id'];

        $tmp    = [static::$items_id => $tID];
        $canadd = $this->can(-1, CREATE, $tmp);
        $rand   = mt_rand();

        if ($canadd) {
            $itemtype = static::$itemtype;
            echo "<form method='post' name=form action='" . $itemtype::getFormURL() . "'>";
        }
        echo "<table class='tab_cadre_fixe'>";
        echo "<tr>";
        echo "<th colspan='3'>" . self::getTypeName(Session::getPluralNumber()) . "</th>";
        echo "</tr>";

        echo "<tr class='tab_bg_1'>";
        echo "<td>" . __('Global approval status') . "</td>";
        echo "<td colspan='2'>";
        echo TicketValidation::getStatus($item->fields["global_validation"], true);
        echo "</td></tr>";

        echo "<tr>";
        echo "<th colspan='2'>" . _x('item', 'State') . "</th>";
        echo "<th colspan='2'>";
        echo self::getValidationStats($tID);
        echo "</th>";
        echo "</tr>";

        echo "<tr class='tab_bg_1'>";
        echo "<td>" . __('Minimum validation required') . "</td>";
        if ($canadd) {
            echo "<td>";
            echo $item->getValueToSelect(
                'validation_percent',
                'validation_percent',
                $item->fields["validation_percent"]
            );
            echo "</td>";
            echo "<td><input type='submit' name='update' class='btn btn-outline-secondary' value='" .
                _sx('button', 'Save') . "'>";
            if (!empty($tID)) {
                echo "<input type='hidden' name='id' value='$tID'>";
            }
            echo "</td>";
        } else {
            echo "<td colspan='2'>";
            echo Dropdown::getValueWithUnit($item->fields["validation_percent"], "%");
            echo "</td>";
        }
        echo "</tr>";
        echo "</table>";
        if ($canadd) {
            Html::closeForm();
        }

        $iterator = $DB->Request([
            'FROM'   => $this->getTable(),
            'WHERE'  => [static::$items_id => $item->getField('id')],
            'ORDER'  => 'submission_date DESC'
        ]);

        $colonnes = [
            '',
            _x('item', 'State'),
            __('Request date'),
            __('Approval requester'),
            __('Request comments'),
            __('Approval status'),
            __('Approver'),
            __('Approval comments'),
            __('Documents')
        ];
        $nb_colonnes = count($colonnes);

        echo "<table class='tab_cadre_fixehov'>";
        echo "<tr class='noHover'><th colspan='" . $nb_colonnes . "'>" . __('Approvals for the ticket') .
            "</th></tr>";

        if ($canadd) {
            /** @var CommonITILObject $item */
            if (
            !in_array($item->fields['status'], array_merge(
                $item->getSolvedStatusArray(),
                $item->getClosedStatusArray()
            ))
            ) {
                echo "<tr class='tab_bg_1 noHover'><td class='center' colspan='" . $nb_colonnes . "'>";
                echo "<a class='btn btn-outline-secondary' href='javascript:viewAddValidation" . $tID . "$rand();'>";
                echo __('Send an approval request') . "</a></td></tr>\n";
            }
        }
        if (count($iterator)) {
            $header = "<tr>";
            foreach ($colonnes as $colonne) {
                $header .= "<th>" . $colonne . "</th>";
            }
            $header .= "</tr>";
            echo $header;

            Session::initNavigateListItems(
                $this->getType(),
                //TRANS : %1$s is the itemtype name, %2$s is the name of the item (used for headings of a list)
                sprintf(
                    __('%1$s = %2$s'),
                    $item->getTypeName(1),
                    $item->fields["name"]
                )
            );

            foreach ($iterator as $row) {
                $canedit = $this->canEdit($row["id"]);
                Session::addToNavigateListItems($this->getType(), $row["id"]);
                $bgcolor = self::getStatusColor($row['status']);
                $status  = self::getStatus($row['status']);

                echo "<tr class='tab_bg_1'>";

                echo "<td>";
                if ($canedit) {
                    echo "<span class='far fa-edit' style='cursor:pointer' title='" . __('Edit') . "' ";
                    echo "onClick=\"viewEditValidation" . $item->fields['id'] . $row["id"] . "$rand();\"";
                    echo " id='viewvalidation" . $this->fields[static::$items_id] . $row["id"] . "$rand'";
                    echo "></span>";
                    echo "\n<script type='text/javascript' >\n";
                    echo "function viewEditValidation" . $item->fields['id'] . $row["id"] . "$rand() {\n";
                    $params = ['type'             => $this->getType(),
                        'parenttype'       => static::$itemtype,
                        static::$items_id  => $this->fields[static::$items_id],
                        'id'               => $row["id"]
                    ];
                    Ajax::updateItemJsCode(
                        "viewvalidation" . $item->fields['id'] . "$rand",
                        $CFG_GLPI["root_doc"] . "/ajax/viewsubitem.php",
                        $params
                    );
                    echo "};";
                    echo "</script>\n";
                }
                echo "</td>";

                echo "<td><div style='background-color:" . $bgcolor . ";'>" . $status . "</div></td>";
                echo "<td>" . Html::convDateTime($row["submission_date"]) . "</td>";
                echo "<td>" . getUserName($row["users_id"]) . "</td>";
                $comment_submission = \Glpi\RichText\RichText::getEnhancedHtml($this->fields['comment_submission'], ['images_gallery' => true]);
                echo "<td><div class='rich_text_container'>" . $comment_submission . "</div></td>";
                echo "<td>" . Html::convDateTime($row["validation_date"]) . "</td>";
                echo "<td>" . getUserName($row["users_id_validate"]) . "</td>";
                $comment_validation = \Glpi\RichText\RichText::getEnhancedHtml($this->fields['comment_validation'] ?? '', ['images_gallery' => true]);
                echo "<td><div class='rich_text_container'>" . $comment_validation . "</div></td>";

                $doc_item = new Document_Item();
                $docs = $doc_item->find(["itemtype"          => $this->getType(),
                    "items_id"           => $this->getID(),
                    "timeline_position"  => ['>', CommonITILObject::NO_TIMELINE]
                ]);
                $out = "";
                foreach ($docs as $docs_values) {
                    $doc = new Document();
                    $doc->getFromDB($docs_values['documents_id']);
                    $out  .= "<a ";
                    $out .= "href=\"" . Document::getFormURLWithID($docs_values['documents_id']) . "\">";
                    $out .= $doc->getField('name') . "</a><br>";
                }
                echo "<td>" . $out . "</td>";

                echo "</tr>";
            }
            echo $header;
        } else {
            //echo "<div class='center b'>".__('No item found')."</div>";
            echo "<tr class='tab_bg_1 noHover'><th colspan='" . $nb_colonnes . "'>";
            echo __('No item found') . "</th></tr>\n";
        }
        echo "</table>";



//        echo "<table class='tab_cadre_fixe mt-4'>";
//        echo "<tr>";
//        echo "<th colspan='3'>" . "Validation pour les tâches" . "</th>";
//        echo "</tr>";
//        echo "</table>";

        echo "<table class='tab_cadre_fixehov mt-4'>";
//        echo "<tr class='noHover'><th colspan='" . $nb_colonnes . "'>" . __('Approvals for the ticket') .
//            "</th></tr>";

        echo "<tr>";
        echo "<th colspan='3'>" . "Contrôle des tâches" . "</th>";
        echo "</tr>";


//        echo "<tr class='tab_bg_1 noHover'><td class='center' colspan='" . $nb_colonnes . "'>";
//        echo "<a class='btn btn-outline-secondary' href='javascript:viewAddValidationForTask" . $tID . "$rand();'>";
//        echo __('Envoyer une demande de contrôle') . "</a></td></tr>\n";


        $tickettasks_forticket_query = [
            "FROM" => TicketTask::getTable(),
            "WHERE" => [
                "tickets_id" => $item->getField('id')
            ]
        ];
//        var_dump("zzz");
//        die();
        $tickettasks_forticket_iterator = $DB->request($tickettasks_forticket_query);
        $tickettasks_idx = array_column(iterator_to_array($tickettasks_forticket_iterator), 'id');
        if(count($tickettasks_idx)>0){
            $iterator = $DB->Request([
                'FROM'   => PluginDlteamsTicketTask_Validation::getTable(),
                'WHERE'  => [
                    "tickettasks_id" => $tickettasks_idx
                ],
                'ORDER'  => 'submission_date DESC'
            ]);

            $colonnes = [
                '',
                _x('item', 'State'),
                __('Request date'),
                __('Actor'),
//                __('Request comments'),
                __('Approval status'),
                __('Approver'), __('Approval comments'), __('Documents')
            ];

            if (count($iterator)) {
                $header = "<tr>";
                foreach ($colonnes as $colonne) {
                    $header .= "<th>" . $colonne . "</th>";
                }
                $header .= "</tr>";
                echo $header;

                Session::initNavigateListItems(
                    $this->getType(),
                    //TRANS : %1$s is the itemtype name, %2$s is the name of the item (used for headings of a list)
                    sprintf(
                        __('%1$s = %2$s'),
                        $item->getTypeName(1),
                        $item->fields["name"]
                    )
                );

                foreach ($iterator as $row) {
                    $canedit = $this->canEdit($row["id"]);
                    Session::addToNavigateListItems($this->getType(), $row["id"]);
                    $bgcolor = self::getStatusColor($row['status']);
                    $status  = self::getStatus($row['status']);

                    echo "<tr class='tab_bg_1'>";

                    echo "<td>";
                    if ($canedit) {
                        echo "<span class='far fa-edit' style='cursor:pointer' title='" . __('Edit') . "' ";
                        echo "onClick=\"viewEditValidation" . $item->fields['id'] . $row["id"] . "$rand();\"";
                        echo " id='viewvalidation" . $this->fields[static::$items_id] . $row["id"] . "$rand'";
                        echo "></span>";
                        echo "\n<script type='text/javascript' >\n";
                        echo "function viewEditValidation" . $item->fields['id'] . $row["id"] . "$rand() {\n";
                        $params = ['type'             => $this->getType(),
                            'parenttype'       => static::$itemtype,
                            static::$items_id  => $this->fields[static::$items_id],
                            'id'               => $row["id"]
                        ];
                        Ajax::updateItemJsCode(
                            "viewvalidation" . $item->fields['id'] . "$rand",
                            $CFG_GLPI["root_doc"] . "/ajax/viewsubitem.php",
                            $params
                        );
                        echo "};";
                        echo "</script>\n";
                    }
                    echo "</td>";


                    echo "<td><div style='background-color:" . $bgcolor . ";'>" . $status . "</div></td>";
                    echo "<td>" . Html::convDateTime($row["submission_date"]) . "</td>";
                    echo "<td>" . getUserName($row["users_id"]) . "</td>";
                    $comment_submission = \Glpi\RichText\RichText::getEnhancedHtml($this->fields['comment_submission'], ['images_gallery' => true]);
//                    echo "<td><div class='rich_text_container'>" . $comment_submission . "</div></td>";
                    echo "<td>" . Html::convDateTime($row["validation_date"]) . "</td>";
                    echo "<td>" . getUserName($row["users_id_validate"]) . "</td>";
                    $comment_validation = \Glpi\RichText\RichText::getEnhancedHtml($this->fields['comment_validation'] ?? '', ['images_gallery' => true]);
                    echo "<td><div class='rich_text_container'>" . $comment_validation . "</div></td>";

                    $doc_item = new Document_Item();
                    $docs = $doc_item->find(["itemtype"          => $this->getType(),
                        "items_id"           => $this->getID(),
                        "timeline_position"  => ['>', CommonITILObject::NO_TIMELINE]
                    ]);
                    $out = "";
                    foreach ($docs as $docs_values) {
                        $doc = new Document();
                        $doc->getFromDB($docs_values['documents_id']);
                        $out  .= "<a ";
                        $out .= "href=\"" . Document::getFormURLWithID($docs_values['documents_id']) . "\">";
                        $out .= $doc->getField('name') . "</a><br>";
                    }
                    echo "<td>" . $out . "</td>";

                    echo "</tr>";
                }
//                echo $header;
            }
            else {
                //echo "<div class='center b'>".__('No item found')."</div>";
                echo "<tr class='tab_bg_1 noHover'><th colspan='" . $nb_colonnes . "'>";
                echo __('No item found') . "</th></tr>\n";
            }
        }
        else {
            //echo "<div class='center b'>".__('No item found')."</div>";
            echo "<tr class='tab_bg_1 noHover'><th colspan='" . $nb_colonnes . "'>";
            echo __('No item found') . "</th></tr>\n";
        }
        echo "</table>";

        echo "<div id='viewvalidation" . $tID . "$rand'></div>\n";

        if ($canadd) {
            echo "<script type='text/javascript' >\n";
            echo "function viewAddValidation" . $tID . "$rand() {\n";
            $params = ['type'             => $this->getType(),
                'parenttype'       => static::$itemtype,
                static::$items_id  => $tID,
                'id'               => -1
            ];
            Ajax::updateItemJsCode(
                "viewvalidation" . $tID . "$rand",
                $CFG_GLPI["root_doc"] . "/ajax/viewsubitem.php",
                $params
            );
            echo "};";
            echo "</script>";
        }


//        dlteams
        if ($canadd) {
            echo "<script type='text/javascript' >\n";
            echo "function viewAddValidationForTask" . $tID . "$rand() {\n";
            $params = ['type'             => PluginDlteamsTicketTask_Validation::getType(),
                'parenttype'       => TicketTask::getType(),
                static::$items_id  => $tID,
                'id'               => -1
            ];
            Ajax::updateItemJsCode(
                "viewvalidation" . $tID . "$rand",
                $CFG_GLPI["root_doc"] . "/marketplace/dlteams/ajax/viewsubitem_fortask.php",
                $params
            );
            echo "};";
            echo "</script>";
        }
    }
}
