<?php

/**
 * ---------------------------------------------------------------------
 *
 * GLPI - Gestionnaire Libre de Parc Informatique
 *
 * http://glpi-project.org
 *
 * @copyright 2015-2025 Teclib' and contributors.
 * @copyright 2003-2014 by the INDEPNET Development Team.
 * @licence   https://www.gnu.org/licenses/gpl-3.0.html
 *
 * ---------------------------------------------------------------------
 *
 * LICENSE
 *
 * This file is part of GLPI.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * ---------------------------------------------------------------------
 */

class TicketTask extends CommonITILTask
{
    public static $rightname = 'task';


    public static function getTypeName($nb = 0)
    {
        return _n('Ticket task', 'Ticket tasks', $nb);
    }


    public static function canCreate()
    {
        return (Session::haveRight(self::$rightname, parent::ADDALLITEM)
            || Session::haveRight('ticket', Ticket::OWN));
    }


    public static function canView()
    {
        return (Session::haveRightsOr(self::$rightname, [parent::SEEPUBLIC, parent::SEEPRIVATE])
            || Session::haveRight('ticket', Ticket::OWN));
    }


    public static function canUpdate()
    {
        return (Session::haveRight(self::$rightname, parent::UPDATEALL)
            || Session::haveRight('ticket', Ticket::OWN));
    }


    public function canViewPrivates()
    {
        return Session::haveRight(self::$rightname, parent::SEEPRIVATE);
    }


    public function canEditAll()
    {
        return Session::haveRight(self::$rightname, parent::UPDATEALL);
    }


    /**
     * Does current user have right to show the current task?
     *
     * @return boolean
     **/
    public function canViewItem()
    {

        if (!$this->canReadITILItem()) {
            return false;
        }

        if (Session::haveRight(self::$rightname, parent::SEEPRIVATE)) {
            return true;
        }

        if (
            !$this->fields['is_private']
            && Session::haveRight(self::$rightname, parent::SEEPUBLIC)
        ) {
            return true;
        }

        // see task created or affected to me
        if (
            Session::getCurrentInterface() == "central"
            && ($this->fields["users_id"] === Session::getLoginUserID())
            || ($this->fields["users_id_tech"] === Session::getLoginUserID())
        ) {
            return true;
        }

        if (
            $this->fields["groups_id_tech"] && ($this->fields["groups_id_tech"] > 0)
            && isset($_SESSION["glpigroups"])
            && in_array($this->fields["groups_id_tech"], $_SESSION["glpigroups"])
        ) {
            return true;
        }

        return false;
    }


    /**
     * Does current user have right to create the current task?
     *
     * @return boolean
     **/
    public function canCreateItem()
    {

        if (!$this->canReadITILItem()) {
            return false;
        }

        $ticket = new Ticket();
        if (
            $ticket->getFromDB($this->fields['tickets_id'])
            // No validation for closed tickets
            && !in_array($ticket->fields['status'], $ticket->getClosedStatusArray())
        ) {
            return (Session::haveRight(self::$rightname, parent::ADDALLITEM)
                || $ticket->isUser(CommonITILActor::ASSIGN, Session::getLoginUserID())
                || (isset($_SESSION["glpigroups"])
                    && $ticket->haveAGroup(
                        CommonITILActor::ASSIGN,
                        $_SESSION['glpigroups']
                    )));
        }
        return false;
    }


    /**
     * Does current user have right to update the current task?
     *
     * @return boolean
     **/
    public function canUpdateItem()
    {

        if (!$this->canReadITILItem()) {
            return false;
        }

        $ticket = new Ticket();
        if (
            $ticket->getFromDB($this->fields['tickets_id'])
            && in_array($ticket->fields['status'], $ticket->getClosedStatusArray())
        ) {
            return false;
        }

        if (
            ($this->fields["users_id"] != Session::getLoginUserID())
            && !Session::haveRight(self::$rightname, parent::UPDATEALL)
        ) {
            return false;
        }

        return true;
    }

//    fork dlteams
    public function post_addItem()
    {
        if(isset($_POST["groups_id_tech"]) && $_POST["groups_id_tech"]){
            $planing_data = $this->fields;
            unset($planing_data["id"]);
            unset($planing_data["uuid"]);
            unset($planing_data["users_id"]);
            unset($planing_data["users_id_tech"]);
            unset($planing_data["groups_id_tech"]);


            $group = new Group();
            $group->getFromDB($_POST["groups_id"]);
            $crit    = Session::getSavedOption(__CLASS__, 'criterion', '');
            $tree    = Session::getSavedOption(__CLASS__, 'tree', 0);
            $members    = [];
            $ids     = [];

            Group_User::getDataForGroup($group, $members, $ids, $crit, $tree, false, true);

            if(count($members)>0) {
                echo "<div class='timeline-badges' style='display: flex; gap: 0.5rem; flex-wrap: wrap'>";
                foreach ($members as $user) {

                    $tickettask = new TicketTask();
                    $tickettask->add([
                        ...$planing_data,
                        "users_id" => $user["id"],
                        "users_id_tech" => $user["id"],
                        "tickettasks_id" => $this->fields["id"]
                    ]);
                }
            }
        }

        parent::post_addItem();
    }

//    fork dlteams
    public function post_updateItem($history = true)
    {
        if(isset($this->oldvalues["state"]) && $this->oldvalues["state"] == Planning::TODO && $this->fields["state"] == Planning::DONE){
            $tickettask_validation = new PluginDlteamsTicketTask_Validation();
            $can_make_task_control = PluginDlteamsTicketTask::areAllPlanificationsDone($this->fields["id"]) && $tickettask_validation->getFromDBByCrit([
                    "tickettasks_id" => $this->fields["id"],
                    "users_id_validate" => 0,
                    "status" => CommonITILValidation::WAITING
                ]);

            if($can_make_task_control){
                $message = new PluginDlteamsMessage();
                $message->add([
                    "itemtype" => PluginDlteamsTicketTask::class,
                    "items_id" => $this->fields["tickettasks_id"]??$this->fields["id"],
                    "content" => "<p>Contrôle de tache : le <b>".date('d/m/Y H:i')."</b> à valider</p>",
                    "entities_id" => Session::getActiveEntity()
                ]);
            }
        }
        parent::post_updateItem($history);
    }


    /**
     * Does current user have right to purge the current task?
     *
     * @return boolean
     **/
    public function canPurgeItem()
    {
        $ticket = new Ticket();
        if (
            $ticket->getFromDB($this->fields['tickets_id'])
            && in_array($ticket->fields['status'], $ticket->getClosedStatusArray())
        ) {
            return false;
        }

        return Session::haveRight(self::$rightname, PURGE);
    }


    /**
     * Populate the planning with planned ticket tasks
     *
     * @param $options   array of possible options:
     *    - who          ID of the user (0 = undefined)
     *    - whogroup     ID of the group of users (0 = undefined)
     *    - begin        Date
     *    - end          Date
     *
     * @return array of planning item
     **/
    public static function populatePlanning($options = []): array
    {
        return self::genericPopulatePlanning(__CLASS__, $options);
    }

    /**
     * Populate the planning with planned tasks
     *
     * @param string $itemtype itemtype
     * @param array $options   options must contains :
     *    - who                ID of the user (0 = undefined)
     *    - whogroup           ID of the group of users (0 = undefined)
     *    - begin              Date
     *    - end                Date
     *    - color
     *    - event_type_color
     *    - display_done_events (boolean)
     *
     * @return array of planning item
     **/
    public static function genericPopulatePlanning($itemtype, $options = [])
    {
        /**
         * @var array $CFG_GLPI
         * @var \DBmysql $DB
         */
        global $CFG_GLPI, $DB;

        $interv = [];

        if (
            !isset($options['begin']) || ($options['begin'] == 'NULL')
            || !isset($options['end']) || ($options['end'] == 'NULL')
        ) {
            return $interv;
        }

        if (!$item = getItemForItemtype($itemtype)) {
            return;
        }
        $parentitemtype = $item->getItilObjectItemType();
        if (!$parentitem = getItemForItemtype($parentitemtype)) {
            return;
        }

        $default_options = [
            'genical'             => false,
            'color'               => '',
            'event_type_color'    => '',
            'display_done_events' => true,
        ];
        $options = array_merge($default_options, $options);

        $who      = $options['who'];
        $whogroup = $options['whogroup']; // direct group
        $begin    = $options['begin'];
        $end      = $options['end'];

        $SELECT = [$item->getTable() . '.*'];

        // Get items to print
        if (isset($options['not_planned'])) {
            //not planned case
            // as we consider that people often create tasks after their execution
            // begin date is task date minus duration
            // and end date is task date
            $bdate = "DATE_SUB(" . $DB->quoteName($item->getTable() . '.date') .
                ", INTERVAL " . $DB->quoteName($item->getTable() . '.actiontime') . " SECOND)";
            $SELECT[] = new QueryExpression($bdate . ' AS ' . $DB->quoteName('notp_date'));
            $edate = $DB->quoteName($item->getTable() . '.date');
            $SELECT[] = new QueryExpression($edate . ' AS ' . $DB->quoteName('notp_edate'));
            $WHERE = [
                $item->getTable() . '.end'     => null,
                $item->getTable() . '.begin'   => null,
                $item->getTable() . '.actiontime' => ['>', 0],
                //begin is replaced with creation tim minus duration
                new QueryExpression($edate . " >= '" . $begin . "'"),
                new QueryExpression($bdate . " <= '" . $end . "'")
            ];
        } else {
            //std case: get tasks for current view dates
            $WHERE = [
                $item->getTable() . '.end'     => ['>=', $begin],
                $item->getTable() . '.begin'   => ['<=', $end]
            ];
        }
        $ADDWHERE = [];

        if ($whogroup === "mine") {
            if (isset($_SESSION['glpigroups'])) {
                $whogroup = $_SESSION['glpigroups'];
            } else if ($who > 0) {
                $whogroup = array_column(Group_User::getUserGroups($who), 'id');
            }
        }

        if ($who > 0) {
            $ADDWHERE[$item->getTable() . '.users_id_tech'] = $who;
        }

        //This means we can pass 2 groups here, not sure this is expected. Not documented :/
        if ($whogroup > 0) {
            $ADDWHERE[$item->getTable() . '.groups_id_tech'] = $whogroup;
        }

        if (!count($ADDWHERE)) {
            $ADDWHERE = [
                $item->getTable() . '.users_id_tech' => new \QuerySubQuery([
                    'SELECT'          => 'glpi_profiles_users.users_id',
                    'DISTINCT'        => true,
                    'FROM'            => 'glpi_profiles',
                    'LEFT JOIN'       => [
                        'glpi_profiles_users'   => [
                            'ON' => [
                                'glpi_profiles_users' => 'profiles_id',
                                'glpi_profiles'       => 'id'
                            ]
                        ]
                    ],
                    'WHERE'           => [
                            'glpi_profiles.interface'  => 'central'
                        ] + getEntitiesRestrictCriteria('glpi_profiles_users', '', $_SESSION['glpiactive_entity'], 1)
                ])
            ];
        }

        if (count($ADDWHERE) > 0) {
            $WHERE[] = ['OR' => $ADDWHERE];
        }

        if (!$options['display_done_events']) {
            $WHERE[] = ['OR' => [
                $item->getTable() . ".state"  => Planning::TODO,
                [
                    'AND' => [
                        $item->getTable() . '.state'  => Planning::INFO,
                        $item->getTable() . '.end'    => ['>', new \QueryExpression('NOW()')]
                    ]
                ]
            ]
            ];
        }

        if ($parentitem->maybeDeleted()) {
            $WHERE[$parentitem->getTable() . '.is_deleted'] = 0;
        }

        if (!$options['display_done_events']) {
            $WHERE[] = ['NOT' => [
                $parentitem->getTable() . '.status' => array_merge(
                    $parentitem->getSolvedStatusArray(),
                    $parentitem->getClosedStatusArray()
                )
            ]
            ];
        }

        $iterator = $DB->request([
            'SELECT'       => $SELECT,
            'FROM'         => $item->getTable(),
            'INNER JOIN'   => [
                $parentitem->getTable() => [
                    'ON' => [
                        $parentitem->getTable() => 'id',
                        $item->getTable()       => $parentitem->getForeignKeyField()
                    ]
                ]
            ],
            'WHERE'        => $WHERE,
            'ORDERBY'      => $item->getTable() . '.begin'
        ]);

        $interv = [];

        if (count($iterator)) {
            foreach ($iterator as $data) {
                if (
                    $item->getFromDB($data["id"])
                    && $item->canViewItem()
                ) {
                    if ($parentitem->getFromDBwithData($item->fields[$parentitem->getForeignKeyField()])) {
                        //not planned
                        if (isset($data['notp_date'])) {
                            $data['begin'] = $data['notp_date'];
                            $data['end'] = $data['notp_edate'];
                        }
                        $key = $data["begin"] .
                            "$$$" . $itemtype .
                            "$$$" . $data["id"] .
                            "$$$" . $who . "$$$" . $whogroup;

                        if (isset($options['from_group_users'])) {
                            $key .= "_gu";
                        }

                        $interv[$key]['color']            = $options['color'];
                        $interv[$key]['event_type_color'] = $options['event_type_color'];
                        $interv[$key]['itemtype']         = $itemtype;
                        $url_id = $item->fields[$parentitem->getForeignKeyField()];
                        if (!$options['genical']) {
                            $interv[$key]["url"] = $parentitemtype::getFormURLWithID($url_id);
                        } else {
                            $interv[$key]["url"] = $CFG_GLPI["url_base"] .
                                $parentitemtype::getFormURLWithID($url_id, false);
                        }
                        $interv[$key]["ajaxurl"] = $CFG_GLPI["root_doc"] . "/ajax/planning.php" .
                            "?action=edit_event_form" .
                            "&itemtype=" . $itemtype .
                            "&parentitemtype=" . $parentitemtype .
                            "&parentid=" . $item->fields[$parentitem->getForeignKeyField()] .
                            "&id=" . $data['id'] .
                            "&url=" . $interv[$key]["url"];

                        $interv[$key][$item->getForeignKeyField()] = $data["id"];
                        $interv[$key]["id"]                        = $data["id"];
                        if (isset($data["state"])) {
                            $interv[$key]["state"]                  = $data["state"];
                        }
                        $interv[$key][$parentitem->getForeignKeyField()]
                            = $item->fields[$parentitem->getForeignKeyField()];
                        $interv[$key]["users_id"]       = $data["users_id"];
                        $interv[$key]["users_id_tech"]  = $data["users_id_tech"];
                        $interv[$key]["groups_id_tech"]  = $data["groups_id_tech"];

                        if (strcmp($begin, $data["begin"]) > 0) {
                            $interv[$key]["begin"] = $begin;
                        } else {
                            $interv[$key]["begin"] = $data["begin"];
                        }

                        if (strcmp($end, $data["end"]) < 0) {
                            $interv[$key]["end"] = $end;
                        } else {
                            $interv[$key]["end"] = $data["end"];
                        }

                        $interv[$key]["name"]     = \Glpi\Toolbox\Sanitizer::unsanitize($parentitem->fields['name']); // name is re-encoded on JS side
                        $interv[$key]["content"]  = \Glpi\RichText\RichText::getSafeHtml($item->fields['content']);
                        $interv[$key]["status"]   = $parentitem->fields["status"];
                        $interv[$key]["priority"] = $parentitem->fields["priority"];

                        $interv[$key]["editable"] = $item->canUpdateITILItem();

                        /// Specific for tickets
                        $interv[$key]["device"] = [];
                        if (
                            $parentitem instanceof Ticket
                            && isset($parentitem->hardwaredatas)
                            && !empty($parentitem->hardwaredatas)
                        ) {
                            foreach ($parentitem->hardwaredatas as $hardwaredata) {
                                $interv[$key]["device"][$hardwaredata->fields['id']] = ($hardwaredata
                                    ? $hardwaredata->getName() : '');
                            }
                            if (is_array($interv[$key]["device"])) {
                                $interv[$key]["device"] = implode("<br>", $interv[$key]["device"]);
                            }
                        }
                    }
                }
            }
        }
        return $interv;
    }

    public function pre_addInDB()
    {

        parent::pre_addInDB(); // TODO: Change the autogenerated stub
    }


    /**
     * Populate the planning with planned ticket tasks
     *
     * @param $options   array of possible options:
     *    - who          ID of the user (0 = undefined)
     *    - whogroup     ID of the group of users (0 = undefined)
     *    - begin        Date
     *    - end          Date
     *
     * @return array of planning item
     **/
    public static function populateNotPlanned($options = []): array
    {
        return parent::genericPopulateNotPlanned(__CLASS__, $options);
    }


    /**
     * Display a Planning Item
     *
     * @param array           $val       array of the item to display
     * @param integer         $who       ID of the user (0 if all)
     * @param string          $type      position of the item in the time block (in, through, begin or end)
     * @param integer|boolean $complete  complete display (more details)
     *
     * @return string
     */
    public static function displayPlanningItem(array $val, $who, $type = "", $complete = 0)
    {
//        todo dlteams
        return static::genericDisplayPlanningItem(__CLASS__, $val, $who, $type, $complete);
    }

//    todo dlteams
    /**
     * Display a Planning Item
     *
     * @param string          $itemtype  itemtype
     * @param array           $val       the item to display
     * @param integer         $who       ID of the user (0 if all)
     * @param string          $type      position of the item in the time block (in, through, begin or end)
     * @param integer|boolean $complete  complete display (more details) (default 0)
     *
     * @return string Output
     **/
    public static function genericDisplayPlanningItem($itemtype, array $val, $who, $type = "", $complete = 0)
    {
        /** @var array $CFG_GLPI */
        global $CFG_GLPI;

        $html = "";
        $rand      = mt_rand();
        $styleText = "";
        if (isset($val["state"])) {
            switch ($val["state"]) {
                case 2: // Done
                    $styleText = "color:#747474;";
                    break;
            }
        }

        $parenttype = str_replace('Task', '', $itemtype);
        if ($parent = getItemForItemtype($parenttype)) {
            $parenttype_fk = $parent->getForeignKeyField();
        } else {
            return;
        }

        // icon Tache

//        todo dlteams, add tickettask id on planning
//        if($tickettask->getType() == TicketTask::getType())
        if (isset($val["state"])) {
            $html .= "<span>";
//            TODO Dlteams
            $status_class = Planning::getStatusClass($val["state"]);
            $status_color = Planning::getStatusColor($val["state"]);
            $status_text = "<b>Tâche</b>&nbsp;".sprintf("<b><a href='/marketplace/dlteams/front/tickettask.form.php?id=%s' target='_blank'>%s</a></b> &nbsp;", $val['id'], $val['id']);
            $status_icon = $status_text."<i class='itilstatus $status_class $status_color me-1' data-bs-toggle='tooltip' data-bs-original-title='Statut' aria-label='Statut'></i>";
            $html .= $status_icon;
            $html .= "</span>";
        }

        $html .= " - ";
//            .$parent->getStatusIcon($val['status']);

        // Icon ticket
        $icon_ticket = "<img src='" . $CFG_GLPI["root_doc"] . "/pics/rdv_interv.png' alt='' title=\"" . Html::entities_deep($parent->getTypeName(1)) . "\">&nbsp;&nbsp;";
        $html .= $icon_ticket."".$parent->getStatusIcon($val['status']);

        // Icon etat ticket
        $tickettask = new TicketTask();
        if($complete ){ // if complet = true mean tooltip
            if($tickettask->getFromDB($val['id']) && $tickettask->fields["tickettasks_id"]){
                $parenttask = new TicketTask();
                $parenttask->getFromDB($tickettask->fields["tickettasks_id"]);
                $html .= htmlspecialchars_decode($parenttask->fields["content"]);
            }
//            $html .= sprintf("<b> du ticket <a href='/marketplace/dlteams/front/ticket.form.php?id=%s' target='_blank'>%s: </a></b>", $tickettask->fields["tickets_id"], $tickettask->fields["tickets_id"]);
        }
        else{
            if($tickettask->getFromDB($val['id']) && $tickettask->fields["tickettasks_id"]){
                $html .= sprintf("( <b> suite de <a href='/marketplace/dlteams/front/tickettask.form.php?id=%s' target='_blank'>%s</a></b> )", $tickettask->fields["tickettasks_id"], $tickettask->fields["tickettasks_id"]);
            }
            if(!$tickettask->fields["tickettasks_id"] && !PluginDlteamsTicketTask::areAllPlanificationsDone($tickettask->fields["id"])){

//                $html.="&nbsp;<i class='itilstatus waiting ti ti-alert-square-filled me-1' data-bs-toggle='tooltip' data-bs-original-title='Des planif. restent A faire' aria-label='Des planif. restent '></i>";
            }
        }

        $html .= sprintf("<b><a href='/front/ticket.form.php?id=%s' target='_blank'>%s </a></b>", $tickettask->fields["tickets_id"], $tickettask->fields["tickets_id"]);

        $html .= "&nbsp;<a id='content_tracking_" . $val["id"] . $rand . "'
                   href='" . $parenttype::getFormURLWithID($val[$parenttype_fk]) . "'
                   style='$styleText'>";

        if (!empty($val["device"])) {
            $html .= "<br>" . $val["device"];
        }

        if ($who <= 0) { // show tech for "show all and show group"
            $html .= "<br>";
            //TRANS: %s is user name
            $html .= sprintf(__('By %s'), getUserName($val["users_id_tech"]));
        }

        $html .= "</a>";

        $recall = '';
        if (
            isset($val[getForeignKeyFieldForItemType($itemtype)])
            && PlanningRecall::isAvailable()
        ) {
            $pr = new PlanningRecall();
            if (
            $pr->getFromDBForItemAndUser(
                $val['itemtype'],
                $val[getForeignKeyFieldForItemType($itemtype)],
                Session::getLoginUserID()
            )
            ) {
                $recall = "<span class='b'>" . sprintf(
                        __('Recall on %s'),
                        Html::convDateTime($pr->fields['when'])
                    ) .
                    "<span>";
            }
        }

        $tickettask_validation = new PluginDlteamsTicketTask_Validation();
        $controles = PluginDlteamsTicketTask::getControles($tickettask->fields["id"]);

//        fork dlteams: block moved



        $html .= "<div>";
//        fork dlteams
//        $html .= sprintf(__('%1$s: %2$s'), __('Priority'), $parent->getPriorityName($val["priority"]));

        $validation_state = "";
        $validation_comment = "";

        if($tickettask_validation->getFromDBByCrit([
            "tickettasks_id" => $tickettask->fields["id"],
            "users_id_validate" => 0
        ])){
            if($tickettask_validation->fields["status"] == CommonITILValidation::ACCEPTED)
                $validation_state = "<span style='color: green'> - ".count($controles["accepted"])." validé(s)</span>";

            if($tickettask_validation->fields["status"] == CommonITILValidation::REFUSED)
                $validation_state = "<span style='color: darkred'> - ".count($controles["refused"])." refusé(s)</span>";

            $validation_comment = htmlspecialchars_decode($tickettask_validation->fields["comment_submission"]??"");
        }


        if($validation_state>0){
            $html .= "<span>";
            $html .= sprintf(__('%s'), $validation_state);
            $html .= "</span>";
        }

        $html .= "</div>";
        //dlteams
//        $validation =
        $can_show_control_end_task = !PluginDlteamsTicketTask::areAllPlanificationsDone($tickettask->fields["id"])
            && $tickettask_validation->getFromDBByCrit([
                "tickettasks_id" => $tickettask->fields["id"],
                "users_id_validate" => 0
            ]);

        $html .= "<div>";
        if($can_show_control_end_task){
            $html .= "<span>";
            $html .= sprintf(__('%3$s %1$s: %2$s'), __('Contrôle(s)'), '"1 attente fin de tâche"', $controles["count"]);
            $html .= "</span>";
        }


        $can_make_task_control = PluginDlteamsTicketTask::areAllPlanificationsDone($tickettask->fields["id"]) && $tickettask_validation->getFromDBByCrit([
                "tickettasks_id" => $tickettask->fields["id"],
                "users_id_validate" => 0,
                "status" => CommonITILValidation::WAITING
            ]);
        if($can_make_task_control){
            $html .= "<span>";
            $html .= sprintf(__(' - %1$s'),count($controles["waiting"]).' "à effectuer"');
            $html .= "</span>";
        }

        $query_tr = [
            "FROM" => ITILFollowup::getTable(),
            "WHERE" => [
                "itemtype" => PluginDlteamsTicketTask::class,
                "items_id" => $tickettask->fields["id"],
                "type" => PluginDlteamsMessage::TICKETTASKREPORT
            ],
            "ORDERBY" => ["id DESC"],
            "LIMIT" => 1
        ];

        global $DB;
        $iterator_report = $DB->request($query_tr);
        if(count(iterator_to_array($iterator_report))>0){
            $report = iterator_to_array($iterator_report)[array_key_first(iterator_to_array($iterator_report))];
            $report["url"] = TicketTask::getFormURLWithID($report["items_id"])."&forcetab=PluginDlteamsMessage$1&#kbcomment".$report["id"];

            $html .= "<span>";
            $html .= " CR: ";
            $html .= "<i class='fa fa-info fa-fw me-1'
                                           title=\"".Glpi\RichText\RichText::getTextFromHtml($report["content"])."\"
                                           data-bs-toggle='tooltip' data-bs-placement='top'></i>";
            $html .= "</span>";
        }




        $html .= "</div>";

        // $val['content'] has already been sanitized and decoded by self::populatePlanning()
        $content = $val['content'];
        $html .= "<div class='event-description rich_text_container'>" . $content . "</div>";
        $html .= $recall;

        return $html;
    }

    /**
     * @since 0.85
     *
     * @see commonDBTM::getRights()
     **/
    public function getRights($interface = 'central')
    {

        $values = parent::getRights();
        unset($values[UPDATE], $values[CREATE], $values[READ]);

        if ($interface == 'central') {
            $values[parent::UPDATEALL]      = __('Update all');
            $values[parent::ADDALLITEM  ]   = __('Add to all items');
            $values[parent::SEEPRIVATE]     = __('See private ones');
        }

        $values[parent::SEEPUBLIC]   = __('See public ones');

        if ($interface == 'helpdesk') {
            unset($values[PURGE]);
        }

        return $values;
    }

//    todo dlteams
    /** form for Task
     *
     * @param $ID        Integer : Id of the task
     * @param $options   array
     *     -  parent Object : the object
     **/
    public function showForm($ID, array $options = [])
    {
//        count planifications
        $planif_query = [
            "FROM" => TicketTask::getTable(),
            "OR" => [
                [
                    "tickettasks_id" => $this->fields["id"]
                ],
                [
                    "id" => $this->fields["id"]
                ]
            ]
        ];
        global $DB;
        $iterator = $DB->request($planif_query);
        $this->fields["countplanif"] = count($iterator);


//        count planifications a faire
        $planif_query = [
            "FROM" => TicketTask::getTable(),
            "OR" => [
                [
                    "tickettasks_id" => $this->fields["id"],
                    "state" => Planning::TODO
                ],
                [
                    "id" => $this->fields["id"],
                    "state" => Planning::TODO
                ]
            ]
        ];
        global $DB;
        $iterator = $DB->request($planif_query);
        $this->fields["countplaniftodo"] = count($iterator);


        $parenttask = new TicketTask();
        $parenttask->getFromDB($this->fields["tickettasks_id"]);
        if(isset($parenttask->fields["content"]))
            $parenttask->fields["content"] = htmlspecialchars_decode($parenttask->fields["content"]);
        \Glpi\Application\View\TemplateRenderer::getInstance()->display('components/itilobject/timeline/form_task.html.twig', [
            'item'               => $options['parent'],
            'subitem'            => $this,
            'parenttask'         => $parenttask,
            'is_subtask'         => !!$this->fields["tickettasks_id"],
            'has_pending_reason' => PendingReason_Item::getForItem($options['parent']) !== false,
            'params'             => $options,
        ]);

        return true;
    }
//    end dlteams

    /**
     * Build parent condition for search
     *
     * @return string
     */
    public static function buildParentCondition()
    {
        return "(0 = 1 " . Ticket::buildCanViewCondition("tickets_id") . ") ";
    }


    public function prepareInputForAdd($input)
    {

        $input["users_id"] = 0;
        // Only set requester on manual action
        if (
            !isset($input['_auto_import'])
            && !isset($input['_auto_update'])
            && !Session::isCron()
        ) {
            $input["users_id"] = Session::getLoginUserID();
        }

//        $input["submission_date"] = $_SESSION["glpi_currenttime"];
//        $input["status"]          = self::WAITING;
//
//        if (!isset($input["users_id_validate"]) || ($input["users_id_validate"] <= 0)) {
//            return false;
//        }
//
//        $itemtype = static::$itemtype;
//        $input['timeline_position'] = $itemtype::getTimelinePosition($input[static::$items_id], $this->getType(), $input["users_id"]);

        return parent::prepareInputForAdd($input);
    }
}
