<?php
/*
 -------------------------------------------------------------------------
 DLTeams plugin for GLPI
 -------------------------------------------------------------------------
 LICENSE : This file is part of DLTeams Plugin.

 DLTeams Plugin is a GNU Free Copylefted software.
 It disallow others people than DLPlace developers to distribute, sell,
 or add additional requirements to this software.
 Though, a limited set of safe added requirements can be allowed, but
 for private or internal usage only ;  without even the implied warranty
 of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

 You should have received a copy of the GNU General Public License
 along with DLTeams Plugin. If not, see <http://www.gnu.org/licenses/>.
 --------------------------------------------------------------------------
  @package   dlteams
  @author    DLPlace developers
  @copyright Copyright (c) 2022 DLPlace
  @inspired	 DPO register plugin (Karhel Tmarr) & gdprropa (Yild)
  @license   GPLv3+ http://www.gnu.org/licenses/gpl.txt
  @link      https://github.com/dlplace/dlteams
  @since     2021
 --------------------------------------------------------------------------
 */

class PluginDlteamsTicketTask extends CommonITILObject
{

    public static $rightname = 'tickettask';
    public $dohistory = true;
    protected $usenotepad = true;

    protected static $showTitleInNavigationHeader = false;

    public function __construct()
    {
        self::forceTable(TicketTask::getTable());
        if (isset($_GET['forcetab'])) {
            Session::setActiveTab(__CLASS__, $_GET['forcetab']);
            unset($_GET['forcetab']);
        }
    }

    public function prepareInputForAdd($input)
    {
        return parent::prepareInputForAdd($input); // TODO: Change the autogenerated stub
    }

    public static function getTaskList($status, $showgrouptickets, $start = null, $limit = null)
    {
        /** @var \DBmysql $DB */
        global $DB;

        $prep_req = ['SELECT' => self::getTable() . '.id', 'FROM' => self::getTable(), 'WHERE' => ["begin" => ["<>", null]]];

        $itemtype = Ticket::class;
        $fk_table = Ticket::getTable();
        $fk_field = Toolbox::strtolower(getPlural($itemtype)) . '_id';

        $prep_req['INNER JOIN'] = [
            $fk_table => [
                'FKEY' => [
                    self::getTable()  => $fk_field,
                    $fk_table         => 'id'
                ]
            ]
        ];


        $prep_req['WHERE'] = [$fk_table . ".status" => $itemtype::getNotSolvedStatusArray()];
        switch ($status) {
            case "todo": // we display the task with the status `todo`
                $prep_req['WHERE'][self::getTable() . '.state'] = Planning::TODO;
                break;
        }
        if ($showgrouptickets) {
            if (isset($_SESSION['glpigroups']) && count($_SESSION['glpigroups'])) {
                $prep_req['WHERE'][self::getTable() . '.groups_id_tech'] = $_SESSION['glpigroups'];
            } else {
                // Return empty iterator result
                $prep_req['WHERE'][] = 0;
            }
        } else {
            $prep_req['WHERE'][self::getTable() . '.users_id_tech'] = $_SESSION['glpiID'];
        }

        $prep_req['WHERE'] += getEntitiesRestrictCriteria($fk_table);
        $prep_req['WHERE'][$fk_table . '.is_deleted'] = 0;

        $prep_req['ORDER'] = [self::getTable() . '.date_mod DESC'];

        if ($start !== null) {
            $prep_req['START'] = $start;
        }
        if ($limit !== null) {
            $prep_req['LIMIT'] = $limit;
        }

        $req = $DB->request($prep_req);
        return $req;
    }

    public static function showCentralList($start, $status = 'todo', $showgrouptickets = true)
    {
        /** @var \DBmysql $DB */
        global $DB;

        $iterator = self::getTaskList($status, $showgrouptickets);

        $total_row_count = count($iterator);
        $displayed_row_count = min((int)$_SESSION['glpidisplay_count_on_home'], $total_row_count);

        if ($total_row_count > 0) {
            $itemtype = get_called_class();
            switch ($status) {
                case "process":
                    $options  = [
                        'reset'    => 'reset',
                        'criteria' => [
                            [
                                'field'      => 12, // status
                                'searchtype' => 'equals',
                                'value'      => 'notold',
                                'link'       => 'AND',
                            ]
                        ],
                    ];
                    if ($showgrouptickets) {
                        $options['criteria'][] = [
                            'field'      => 112, // tech in charge of task
                            'searchtype' => 'equals',
                            'value'      => 'mygroups',
                            'link'       => 'AND',
                        ];
                    } else {
                        $options['criteria'][] = [
                            'field'      => 95, // tech in charge of task
                            'searchtype' => 'equals',
                            'value'      => $_SESSION['glpiID'],
                            'link'       => 'AND',
                        ];
                    }
                    $options['criteria'][] = [
                        'field'      => 33, // task status
                        'searchtype' => 'equals',
                        'value'      =>  Planning::TODO,
                        'link'       => 'AND',
                    ];

                    $title = __("Mes tâches planifiées");
                    $type = Ticket::getTypeName();
                    $parent_itemtype = Ticket::class;

                    $linked_itemtype = str_replace("Task", "", $itemtype);
                    $main_header = "<a href=\"" . $linked_itemtype::getSearchURL() . "?" .
                        Toolbox::append_params($options, '&amp;') . "\">" .
                        Html::makeTitle($title, $displayed_row_count, $total_row_count) . "</a>";
                    break;

                default:
                    // Invalid status
                    return;
            }

            $twig_params = [
                'class'        => 'table table-borderless table-striped table-hover card-table',
                'header_rows'  => [
                    [
                        [
                            'colspan'   => 3,
                            'content'   => $main_header
                        ]
                    ],
                ],
                'rows'         => []
            ];

            $i = 0;
            if ($displayed_row_count > 0) {
                $twig_params['header_rows'][] = [
                    [
                        'content'   => __('ID'),
                        'style'     => 'width: 75px'
                    ],
                    [
                        'content'   => __('Title') . " (" . strtolower($type) . ")",
                        'style'     => 'width: 20%'
                    ],
                    __('Description')
                ];
                foreach ($iterator as $data) {
                    $row = [
                        'values' => []
                    ];

                    $task = $itemtype::getById($data['id']);
                    $parent_item = $parent_itemtype::getById($task->fields[getForeignKeyFieldForItemType($parent_itemtype)]);


                    if (!$task || !$parent_item) {
                        // Invalid data; skip
                        continue;
                    }

                    // Parent item id with priority hint
                    $bgcolor = $_SESSION["glpipriority_" . $parent_item->fields["priority"]];
                    $name = sprintf(__('%1$s: %2$s'), __('ID'), $parent_item->fields["id"]);
                    $row['values'][] = [
                        'content' => "<div class='priority_block' style='border-color: $bgcolor'><span style='background: $bgcolor'></span>&nbsp;$name</div>"
                    ];

                    // Parent item name
                    $row['values'][] = [
                        'content' => $parent_item->fields['name']
                    ];

                    // Task description
                    $href = $parent_item::getFormURLWithID($parent_item->fields['id']);
                    $link_title = Html::resume_text(\Glpi\RichText\RichText::getTextFromHtml($task->fields['content'], false, true, true), 50);
                    $row['values'][] = [
                        'content' => "<a href='$href'>$link_title</a>"
                    ];

                    $twig_params['rows'][] = $row;

                    $i++;
                    if ($i == $displayed_row_count) {
                        break;
                    }
                }
            }
            echo \Glpi\Application\View\TemplateRenderer::getInstance()->render('components/table.html.twig', $twig_params);
        }
    }

    public function getHeaderName(): string
    {
        $name = "<span class='me-1'>" . $this->getStatusIcon($this->fields['status']) . '</span>';
        return $name.parent::getHeaderName(); // TODO: Change the autogenerated stub
    }

    static function canCreate()
    {
        return true;
    }

    static function canView()
    {
        return true;
    }

    static function canUpdate()
    {
        return true;
    }

    static function canDelete()
    {
        return true;
    }

    static function canPurge()
    {
        return true;
    }

    function canCreateItem()
    {
        return true;
    }

    function canViewItem()
    {
        return true;
    }

    function canUpdateItem()
    {
        return true;
    }

    function canDeleteItem()
    {
        return true;
    }

    function canPurgeItem()
    {
        return true;
    }

    public static function getTypeName($nb = 0)
    {
        return _n('Tâche', 'Tâches', $nb);
    }

    public static function getIcon()
    {
        return 'fas fa-tasks';
    }

    public static function dropdown($options = [])
    {
        /// TODO try to revert usage : Dropdown::show calling this function
        /// TODO use this function instead of Dropdown::show
        return Dropdown::show(get_called_class(), $options);
    }

    public static function getStatusIcon($status)
    {
        return parent::getStatusIcon($status); // TODO: Change the autogenerated stub
    }

    public static function getControles($tickettasks_id)
    {

        $query = [
            "FROM" => PluginDlteamsTicketTask_Validation::getTable(),
            "WHERE" => [
                "tickettasks_id" => $tickettasks_id,
                "users_id_validate" => 0,
            ]
        ];

        global $DB;

        $controles = ["waiting" => [], "refused" => [], "accepted" => []];
        $iterator = $DB->request($query);
        $count = 0;
        foreach ($iterator as $controle) {
            $count++;
            if ($controle["status"] == CommonITILValidation::WAITING)
                $controles["waiting"][] = $controle;
            if ($controle["status"] == CommonITILValidation::REFUSED)
                $controles["refused"][] = $controle;
            if ($controle["status"] == CommonITILValidation::ACCEPTED)
                $controles["accepted"][] = $controle;
        }

        $controles["count"] = $count;
        return $controles;
    }





    public static function areAllPlanificationsDone($tickettasks_id)
    {
        if (!$tickettasks_id) {
            return false;
        }

        $tickettask = new PluginDlteamsTicketTask();

        // Check if ticket task exists and is marked as DONE
        if (!$tickettask->getFromDB($tickettasks_id) || $tickettask->fields["state"] != Planning::DONE) {
            return false;
        }

        global $DB;

        // Check if there are related planifications marked as DONE
        $planif_done = $DB->request([
            "FROM" => TicketTask::getTable(),
            "WHERE" => [
                "tickettasks_id" => $tickettasks_id,
                "state" => Planning::DONE
            ]
        ]);

        // If there are planifications marked as DONE or no related planifications exist
        return count($planif_done) > 0 || !$tickettask->getFromDBByCrit(["tickettasks_id" => $tickettasks_id]);
    }


    /**
     * Get an object using some criteria
     *
     * @param array $crit search criteria
     *
     * @return boolean|array
     * @since 9.2
     *
     */
    public function getFromDBByCrit(array $crit)
    {
        /** @var \DBmysql $DB */
        global $DB;

        $crit = ['SELECT' => 'id',
            'FROM' => $this->getTable(),
            'WHERE' => $crit
        ];

        $iter = $DB->request($crit);
        if (count($iter) == 1) {
            $row = $iter->current();
            return $this->getFromDB($row['id']);
        } else if (count($iter) > 1) {
//            trigger_error(
//                sprintf(
//                    'getFromDBByCrit expects to get one result, %1$s found in query "%2$s".',
//                    count($iter),
//                    $iter->getSql()
//                ),
//                E_USER_WARNING
//            );
        }
        return false;
    }


    public static function canHaveReview($tickettasks_id): bool
    {
        if (!$tickettasks_id)
            return false;
        $tickettask_validation = new PluginDlteamsTicketTask_Validation();
        if (!$tickettask_validation->getFromDBByCrit([
            "tickettasks_id" => $tickettasks_id,
//            "users_id_validate" => 0,
            "status" => PluginDlteamsTicketTask_Validation::ACCEPTED
        ])
//        &&
//            $tickettask_validation->getFromDBByCrit([
//                "tickettasks_id" => $tickettasks_id,
//                "users_id_validate" => 0
//            ])
        ) {
            return true;
        }

        return false;
    }

    public static function getAdditionalMenuLinks()
    {
        $links = [];


        // ajout d'un bouton ajouter une planification -> ne fonctionne pas -> bouton en bas
        /* $image = "<i class='fas fa-file-signature fa-1x' title='" . __('Ajouter une planification', 'dlteams') . "'></i><span>Ajouter une planification</span>";
        $links[$image] = 'add_tickettask_plannification'*/

        if (Planning::canView()) {
            $title = Planning::getTypeName(Session::getPluralNumber());
            $planning = "<i class='fa far fa-calendar-alt pointer' title='$title'>
                       </i><span>Planning</span>";
            $links["$planning"] = Planning::getSearchURL(false);
        }


        echo "<script>
                        \$(document).ready(function () {
                            // Attendre que la page soit complètement chargée
                            \$('a[href^=\"/front/planning.php\"]').attr('target', '_blank');
                        });
                    </script>";
        return $links;
    }

    public function getName($options = [])
    {
        $ticket = new Ticket();
        $ticket->getFromDB($this->fields["tickets_id"]);
        // $text = isset($this->fields["timeline_position"]) ? sprintf('#%s', $this->fields["timeline_position"]) : "";
        $text = isset($this->fields["id"]) ? sprintf("%s ", $this->fields["id"]) : "";
        $text .= isset($ticket->fields["id"]) ? sprintf("\n de l'évènement %s", $this->fields["tickets_id"]) : "";
        return $text; // TODO: Change the autogenerated stub
    }

    public static function getTable($classname = null)
    {
        return TicketTask::getTable();
    }


    public function getTabNameForItem(CommonGLPI $item, $withtemplate = 0)
    {
        switch ($item->getType()) {
            case Ticket::class:
                if (!$withtemplate) {
                    if (Session::haveRight($item::$rightname, READ)) {
                        if ($_SESSION['glpishow_count_on_tabs']) {
                            $count = 0;

                            $timeline = $item->getTimelineItems(["taskswithplanned" => true]);
                            $cumul_temps = 0;
                            foreach ($timeline as $key => $timeline_item) {
                                if ($timeline_item["type"] == TicketTask::class) {
                                    $count++;
                                }
                            }
                            return static::createTabEntry(static::getTypeNameForClass(), $count);
                        }
                        return static::getTypeNameForClass();
                    }
                }
                break;
        }

        return '';
    }


    public static function displayTabContentForItem(CommonGLPI $item, $tabnum = 1, $withtemplate = 0)
    {
        switch ($item->getType()) {
            case Ticket::class:
                self::showForItem($item);
                break;
        }
    }

    public function post_updateItem($history = true)
    {
        if (isset($this->oldvalues["state"]) && $this->oldvalues["state"] == Planning::TODO && $this->fields["state"] == Planning::DONE) {
            $tickettask_validation = new PluginDlteamsTicketTask_Validation();
            $can_show_control_end_task = !PluginDlteamsTicketTask::areAllPlanificationsDone($this->fields["id"])
                && $tickettask_validation->getFromDBByCrit([
                    "tickettasks_id" => $this->fields["id"],
                    "users_id_validate" => 0
                ]);

            if ($can_show_control_end_task) {
                $message = new PluginDlteamsMessage();
                $message->add([
                    "itemtype" => PluginDlteamsTicketTask::class,
                    "items_id" => $this->fields["tickettasks_id"] ?? $this->fields["id"],
                    "content" => "<p>Contrôle de tache : le <b>" . date('d/m/Y H:i:s') . "</b> à valider</p>"
                ]);
            }
        }
    }

    static function showForItem(CommonDBTM $item, $withtemplate = 0)
    {
        $timeline = $item->getTimelineItems([
            "taskswithplanned" => true
        ]);

        $cumul_temps = 0;
        foreach ($timeline as $key => $timeline_item) {
            if ($timeline_item["type"] == TicketTask::class) {
                $cumul_temps += $timeline_item["item"]["actiontime"];

                $timeline[$key]["item"]["date_str"] = date('d/m/Y H:i:s', strtotime($timeline_item["item"]["date"])) ?? "--";
                $group = new Group();
                $group->getFromDB($timeline_item["item"]["groups_id_tech"]);
                $timeline[$key]["item"]["groupe_str"] = isset($group->fields["name"]) ? $group->fields["name"] : "--";

                $user = new User();
                $user->getFromDB($timeline_item["item"]["users_id_tech"]);
                $timeline[$key]["item"]["acteur_str"] = $timeline[$key]["item"]["users_id_tech"] ? sprintf("%s %s", $user->fields["firstname"], $user->fields["realname"]) : "--";
                $timeline[$key]["item"]["duree_str"] = PluginDlteamsTicketTask_Planning::convertirTemps($timeline[$key]["item"]["actiontime"]) ?? "--";
                $timeline_item["item"]["content"] = htmlspecialchars_decode($timeline_item["item"]["content"]);
//                var_dump(htmlspecialchars_decode($timeline_item["item"]["content"]));
//                die();
                if ($timeline_item["item"]["begin"]) {
                    $planif_fin = new DateTime($timeline_item["item"]["begin"]);
                    $planif_fin->modify("+" . $timeline_item["item"]["actiontime"] . " second");
                }
                $timeline[$key]["item"]["plannif_str"] = sprintf("%s %s", $timeline_item["item"]["begin"] ? date('d/m/Y H:i', strtotime($timeline_item["item"]["begin"] ?? "")) : "--",
                    isset($planif_fin) ? " => " . $planif_fin->format('d/m/Y H:i') : "");

            }
        }

        $cumul_formatted = PluginDlteamsTicketTask::convertirTimestampToHour($cumul_temps);
        \Glpi\Application\View\TemplateRenderer::getInstance()->display('@dlteams/components/itilobject/layout.html.twig', [
            'item' => $item,
            'timeline_itemtypes' => $item->getTimelineItemtypes(),
            'legacy_timeline_actions' => $item->getLegacyTimelineActionsHTML(),
            'params' => [],
            'entities_id' => Session::getActiveEntity(),
            'timeline' => $timeline,
            'cumul_formatted' => $cumul_formatted,
            'itiltemplate_key' => Ticket::getTemplateFormFieldName(),
            'itiltemplate' => $item->getITILTemplateToUse(
                $options['template_preview'] ?? 0,
                $item->fields['type'],
                $item->fields['itilcategories_id'],
                $item->fields['entities_id'],
            ),

        ]);
    }

    static function getTypeNameForClass($nb = 0)
    {
        return __("Tâches", 'dlteams');
    }

    public function showForm($ID, array $options = [])
    {

        $tickettask_validation = new PluginDlteamsTicketTask_Validation();
//        $alertKey = 'alertClosed_' . $this->getType() . '_' . $this->fields["id"];
//
//        if (!PluginDlteamsTicketTask::areAllPlanificationsDone($this->fields["id"])
//            && $tickettask_validation->getFromDBByCrit([
//                "tickettasks_id" => $this->fields["id"],
//                "users_id_validate" => 0
//            ])) {
//            echo '<div class="alert alert-warning d-flex justify-content-between align-items-center mx-4 alert-dismissible mt-3" id="planification-alert">
//          <div class="d-flex align-items-center">
//              <i class="fas fa-2x fa-info me-2"></i>
//              <span>Des planifications sont en attente de fin pour le contrôle de cette tâche</span>
//          </div>
//          <button type="button" class="btn-close" aria-label="Close" onclick="closeAlert(\'' . $alertKey . '\')"></button>
//      </div>';
//
//
//
//            echo '
//      <script>
//        $(document).ready(function() {
//            function closeAlert(key) {
//                // Masquer l\'alerte
//                document.getElementById("planification-alert").style.display = "none";
//
//                // Stocker l\'information que l\'alerte a été fermée dans le localStorage
//                localStorage.setItem(key, "true");
//            }
//
//            // Vérifier si l\'alerte a déjà été fermée
//            window.onload = function() {
//                var alertKey = "' . $alertKey . '";
//                if (localStorage.getItem(alertKey) === "true") {
//                    document.getElementById("planification-alert").style.display = "none";
//                }
//            };
//        });
//        </script>';
//                }
        $parent = new TicketTask();
        $parent->getFromDB($this->fields["tickettasks_id"]);
        global $CFG_GLPI;
        $this->initForm($ID, $options);
        $this->showFormHeader($options);

        echo "<input type='hidden' name='itemtype' value='" . Ticket::class . "'>";
        if(isset($_GET["projects_id"]))
            echo "<input type='hidden' name='_projects_id' value='" . $_GET["projects_id"] . "'>";
        echo "<table style='width: 100%'>";
        echo "<tr>";
        echo "<td style='text-align:right'>" . __("Évènement", 'dlteams') . "</td>";

        $rand = mt_rand();
        echo "<td style='text-align:left'>";
        if (!$this->fields["tickettasks_id"]) {
            Ticket::dropdown([
                'name' => 'tickets_id',
                'rand' => $rand,
                'value' => $this->fields["tickets_id"],
                'entity' => Session::getActiveEntity(),
                'width' => '500px',
//                'readonly' => true,
                'url' => $CFG_GLPI['root_doc'] . "/marketplace/dlteams/ajax/getDropdownValue.php"
            ]);
        } else {
            $content = new Ticket();
            $content->getFromDB($this->fields['tickets_id']);
            echo "<a target='_blank' href=\"" . Ticket::getFormURLWithID($this->fields['tickets_id']) . "\">" . $content->getName() . "</a>"; // . " (" . $content->getId() . ")";
        }
        echo "</td>";
        echo "</tr>";


        if ($this->fields["tickettasks_id"]) {

            echo "<tr>";
            echo "<td style='text-align:right'>" . __("Tâche liée", 'dlteams') . "</td>";
            echo "<td style='text-align: left'>";
            echo "<a target='_blank' href=\"" . TicketTask::getFormURLWithID($this->fields['tickettasks_id']) . "\">" . $this->fields['tickettasks_id'] . "</a>"; // . " (" . $content->getId() . ")";
            echo "</td>";
            echo "<td width='30%'>" . " " . "</td>";
            echo "</tr>";
        }


        if ($this->getID()) {
            echo "<tr>";
            echo "<td style='text-align:right'></td>";
            echo "<td style='text-align: left'>"; //. "<td colspan = 3>";

            $entry_i = $this->fields;
            global $DB;

            $nb_planif = count($DB->request([
                "FROM" => TicketTask::getTable(),
                "WHERE" => [
                    "tickettasks_id" => $this->fields["id"]
                ]
            ]));

            $nb_planif_todo = count($DB->request([
                "FROM" => TicketTask::getTable(),
                "OR" => [
                    [
                        "tickettasks_id" => $this->fields["id"],
                        "state" => Planning::TODO
                    ],
                    [
                        "id" => $this->fields["id"],
                        "state" => Planning::TODO
                    ]
                ]
            ]));

            $entry_i["nb_planification"] = $nb_planif + 1; // + 1 en comptant la tache elle meme comme une planification
            $entry_i["nb_planification_todo"] = $nb_planif_todo;

            $entry_i["can_have_review"] = PluginDlteamsTicketTask::canHaveReview($this->fields["id"]);


            $entry_i["can_show_control_end_task"] = !PluginDlteamsTicketTask::areAllPlanificationsDone($this->fields["id"])
                && $tickettask_validation->getFromDBByCrit([
                    "tickettasks_id" => $this->fields["id"],
                    "users_id_validate" => 0
                ]);



            $entry_i["can_make_task_control"] = PluginDlteamsTicketTask::areAllPlanificationsDone($this->fields["id"]) && $tickettask_validation->getFromDBByCrit([
                    "tickettasks_id" => $this->fields["id"],
                    "users_id_validate" => 0,
                    "status" => CommonITILValidation::WAITING
                ]);

            $messages_iterator = $DB->request([
                "FROM" => PluginDlteamsMessage::getTable(),
                "WHERE" => [
                    "items_id" => $this->fields["id"],
                    "itemtype" => PluginDlteamsTicketTask::class
                ],
                "ORDER" => "id DESC"
            ]);
            $entry_i["nb_messages"] = count($messages_iterator);
            $messages = iterator_to_array($messages_iterator, 0);

            if (count($messages) > 0) {
                $entry_i["last_message"] = $messages[0];
                $sender = new User();
                $sender->getFromDB($messages[0]["users_id"]);
                $entry_i["last_message"]["sender"] = $sender->getUserInitials();
            }


            $validation_state = "";
            $validation_comment = "";

            if ($tickettask_validation->getFromDBByCrit([
                "tickettasks_id" => $this->fields["id"],
                "users_id_validate" => 0
            ])) {
                if ($tickettask_validation->fields["status"] == CommonITILValidation::ACCEPTED)
                    $validation_state = "Contrôle: 1 validé";

                if ($tickettask_validation->fields["status"] == CommonITILValidation::REFUSED)
                    $validation_state = "Contrôle: 1 refusé";

                $validation_comment = htmlspecialchars_decode($tickettask_validation->fields["comment_submission"] ?? "");
                $entry_i["validation_id"] = $tickettask_validation->fields["id"];
            }
            $entry_i["validation_state"] = $validation_state;
            $entry_i["validation_comment"] = $validation_comment;
//
//                get validations
            $validations_query = [
                "FROM" => PluginDlteamsTicketTask_Validation::getTable(),
                "WHERE" => [
                    "tickettasks_id" => $this->fields["id"],
                ],
                "ORDER" => ["id desc"]
            ];

            $iterator_validation = $DB->request($validations_query);
            $validations = iterator_to_array($iterator_validation);

            if (count($validations) > 0) {
                $entry_i["validation"] = isset($validations[0]) ? $validations[0] : null;
            }


            if ($entry_i['nb_planification'] > 0) {
                echo '<span><b>Planifications:</b> ' . $entry_i['nb_planification'] . ' </span>';
                if ($entry_i['nb_planification_todo'] > 0) {
                    echo 'dont ' . $entry_i['nb_planification_todo'] . ' "A faire"';
                } else {
                    echo 'terminée(s)';
                }
            }

            if ($entry_i['can_show_control_end_task'] > 0) {
                echo '- <span><b> Contrôle:</b> "1 attendre fin de tâche"</span>';
            }

            if ($entry_i['can_make_task_control'] > 0) {
                echo '- <span><b> Contrôle: "1 à effectuer"</b></span>';
            }

            if (!static::canHaveReview($this->fields["id"])) {
                echo ' et dernier contrôle ok';
            }

            if ($entry_i['nb_messages'] > 0) {
                echo '- <span><b>dernier message:</b> </span> '
                    . date('d/m/Y H:i', strtotime($entry_i['last_message']['date_creation']))
                    . ' par ' . htmlspecialchars($entry_i['last_message']['sender'], ENT_QUOTES, 'UTF-8');
            }

            if ($entry_i['validation_state'] > 0) {
                echo '- <span><b style="cursor: pointer">'
                    . htmlspecialchars($entry_i["validation_state"], ENT_QUOTES, 'UTF-8')
                    . '</b></span>';
            }

            echo "</td>";
            echo "<td width='30%'>" . " " . "</td>";
            echo "</tr>";
        }


        echo "<tr>";
        echo "<td style='text-align:right'>" . __("Contenu", 'dlteams') . "</td>";
        echo "<td style='text-align: left'>"; //. "<td colspan = 3>";
        $content_id = "content$rand";
        if (!$this->fields["tickettasks_id"])
            Html::textarea(['name' => 'content',
                'value' => \Glpi\RichText\RichText::getSafeHtml($this->fields["tickettasks_id"] ? $this->fields['content'] : $this->fields['content'], true),
                'rand' => $rand,
                'editor_id' => $content_id,
                'enable_fileupload' => true,
                'enable_richtext' => true,
                'cols' => 100,
                'rows' => 15
            ]);
        else {
//            echo "<a target='_blank' href=\"" . TicketTask::getFormURLWithID($this->fields['tickettasks_id']) . "\">" . \Glpi\RichText\RichText::getSafeHtml($this->fields['content'], false) . "</a>";
            $parent = new TicketTask();
            $parent->getFromDB($this->fields['tickettasks_id']);
            $sanshtml = \Glpi\RichText\RichText::getSafeHtml($parent->fields['content'], false);
            echo "-------<br> "; // echo "<br>"."(".$parent->fields['id'].") " ;
            echo sprintf("<div> (%s) %s", $parent->fields['id'], $sanshtml . "</div>");
        }
        echo "</td>";
        echo "<td width='30%'>" . " " . "</td>";
        echo "</tr>";
        echo "</table>";

        echo "<table style='width: 60%; margin-left: 33px;'>";
        echo "<tr>";
        echo "<td style='width: 13.5%'>" . __("Gabarit", 'dlteams') . "</td>";
        echo "<td style='text-align: left'>";
        if (!$this->fields["tickettasks_id"])
            TaskTemplate::dropdown([
                'name' => 'tasktemplates_id',
                'rand' => $rand,
                'value' => $this->fields["tasktemplates_id"],
                'entity' => Session::getActiveEntity(),
                'width' => '250px',
            ]);
        else {
            $element = new TaskTemplate();
            if (isset($parent->fields['tasktemplates_id'])) {
                $element->getFromDB($parent->fields['tasktemplates_id']);
                echo "<a target='_blank' href=\"" . TaskTemplate::getFormURLWithID($parent->fields['tasktemplates_id']) . "\">" . $element->getName() . "</a>";
            } else
                echo "--";
        }
        // echo htmlspecialchars($this->fields['tasktemplates_id']);
        echo "</td>";

        echo "<td style=''>" . __("Catégorie", 'dlteams') . "</td>";
        echo "<td style='text-align: left'>";
        if (!$this->fields["tickettasks_id"])
            TaskCategory::dropdown([
                'name' => 'taskcategories_id',
                'rand' => $rand,
                'value' => $this->fields["taskcategories_id"],
                'entity' => Session::getActiveEntity(),
                'width' => '250px'
            ]);
        else {
            if (isset($parent->fields['taskcategories_id'])) {
                $element = new TaskCategory();
                $element->getFromDB($parent->fields['taskcategories_id']);
                echo "<a target='_blank' href=\"" . TaskCategory::getFormURLWithID($parent->fields['taskcategories_id']) . "\">" . $element->getName() . "</a>";
            } else
                echo "--";
        }
        // echo htmlspecialchars($this->fields['taskcategories_id']);
        echo "</td>";
        echo "</tr>";

        echo "<tr>";
        echo "<td style=''>" . __("Acteur", 'dlteams') . "</td>";
        echo "<td style='text-align:left; width: 10%'>";
        User::dropdown([
            'addicon' => true,
            'name' => 'users_id_tech',
            'value' => $this->fields['users_id_tech'],
            'entity' => Session::getActiveEntity(),
            'right' => 'all',
            'width' => '250px',
        ]);
        echo "</td>";
//        echo "</tr>";
//        echo "<tr>";
        echo "<br/>";
        echo "<td style='text-align:right'>" . __("Groupe", 'dlteams') . "</td>";
        echo "<td style='text-align:left'>";
        Group::dropdown([
            'name' => 'groups_id_tech',
            'value' => $this->fields['groups_id_tech'],
            'entity' => Session::getActiveEntity(),
            'right' => 'all',
            'width' => '250px',
        ]);
        echo "</td>";
        echo "</tr>";

        echo "<tr>";
        echo "<td style='text-align:right'>" . __("Échéance", 'dlteams') . "</td>";
        echo "<td style='text-align:left'>";
        // echo "<div style='width: 60%'>";
        Html::showDateTimeField('date', [
            // $timeline[$key]["item"]["date_str"] = date('d/m/Y H:i:s', strtotime($timeline_item["item"]["date"]))??"--";
            'value' => date('d/m/Y H:i', strtotime($this->fields["date"])) != '' ? $this->fields["date"] : '',
            'rand' => $rand,
            'display' => true,
            'datatype' => 'datetime',
            'width' => '200px'
        ]);
        echo "</div>";
        echo "</td>";

        echo "<td style='text-align:right'>" . __("Durée prévue", 'dlteams') . "</td>";
        echo "<td style='text-align:left'  width='100px'>";
        Dropdown::showTimeStamp('estimate_duration', [
            'icon_label' => true,
            'rand' => $rand,
            'min' => 0,
            'max' => 8 * constant('HOUR_TIMESTAMP'),
            'addfirstminutes' => true,
            'inhours' => true,
            'value' => $this->fields["estimate_duration"],
            'toadd' => array_map(function ($i) {
                return $i * HOUR_TIMESTAMP;
            }, range(9, 100)),
        ]);
        echo "</td>";
        echo "</tr>";

        echo "<tr>";
        echo "<td style='text-align:right'></td>";
        echo "<td colspan='3'>";

        $in_modal = isset($options["in_modal"]) ? $options["in_modal"] : false;
        $width = $in_modal ? '100%' : '100%';

        echo "</td>";
        echo "</tr>";
        echo "</table>";

        echo "<div id='viewplanPlanif' ></div>";
        echo "<div style='display: flex; flex-direction: column; gap: 10px; text-align: start'>";

        echo "<div class='center firstbloc' style='display: flex; flex-direction: row; gap: 5px'>";
//        echo "<a class='btn btn-primary' href='#'>" .
//            _x('button', 'Voir planning') . "</a>";


        // Planification de la tache
        if (!$this->fields["begin"]) {
            echo "<table style='width: 20%; margin-top: 20px'>";
            echo "<tr class='tab_bg_2'><td>";
            echo "<button id='planplanif' class='btn btn-outline-secondary text-truncate' type='button'>
                              <i class='fas fa-calendar'></i>
                              <span>" . __('Plan this task') . "</span>
                           </button>";
            echo "</td>";
            echo "</tr>";
            echo "</table>";
        }


        echo "<script>
        $('#periodiquebtn').on('click', function () {
            var comment_id = $(this).attr('data-comment-id');
            var modalId = glpi_ajax_dialog({
                dialogclass: 'modal-xl',
                bs_focus: false,
                url: '/marketplace/dlteams/ajax/recurrenttaskplannif.php',
                params: {
                    tickettasks_id: $ID
                    },
                title: i18n.textdomain('dlregister').__('Rendre cette tâche périodique', 'dlregister'),
                close: function () {
                },
                fail: function () {
                    // displayAjaxMessageAfterRedirect();
                }
            });
            });
        </script>";


        echo "</div>";


        $tickettask_validation = new PluginDlteamsTicketTask_Validation();
        $can_show_control_end_task = !PluginDlteamsTicketTask::areAllPlanificationsDone($this->fields["id"])
            && $tickettask_validation->getFromDBByCrit([
                "tickettasks_id" => $this->fields["id"],
                "users_id_validate" => 0
            ]);

        $can_make_task_control = PluginDlteamsTicketTask::areAllPlanificationsDone($this->fields["id"]) && $tickettask_validation->getFromDBByCrit([
                "tickettasks_id" => $this->fields["id"],
                "users_id_validate" => 0,
                "status" => CommonITILValidation::WAITING
            ]);


        // une fois planifié, on peut en ajouter d'autres
        echo "<table style='width:40%; margin-left: 35px;'>";

        if ($this->getID() && !$this->fields["tickettasks_id"]) {
            echo "<td align='center'>";
            echo "<button name='add_report' id='add_report' style='width: fit-content;' class='btn btn-primary makecontrol'>" .
                _x('button', 'Ajouter un compte rendu') . "</button>";
            echo "</td>";
        }

        if ($can_make_task_control) {
            echo "<td align='center'>";
            echo "<button name='makecontrol' id='makecontrol' style='width: fit-content;' class='btn btn-primary makecontrol'>" .
                _x('button', 'Effectuer l\'évaluation') . "</button>";
            echo "</td>";
        }


        if (!$this->fields["tickettasks_id"] && $this->fields["begin"]) {
            echo "<td align='center'>";
            echo "<button name='add_tickettask_plannification' id='add_tickettask_plannification' style='width: fit-content;' class='btn btn-primary'>" .
                _x('button', 'Ajouter planification') . "</button>";
            echo "</td>";

        }

        if ($this->fields["id"] && !$this->fields["tickettasks_id"]) {
            echo "<td align='center'>";
            echo "<button id='periodiquebtn' class='btn btn-outline-secondary text-truncate' type='button'>
                              <i class='fas fa-calendar-time'></i>
                              <span>" . __('Rendre périodique') . "</span>
                           </button>";
            echo "</td>";
        }


//        var_dump(static::canHaveReview($this->fields["id"]));
//        die();
        if (!$can_make_task_control && static::canHaveReview($this->fields["id"]) && !$this->fields["tickettasks_id"]) {
            echo "<td align='center'>";
            echo "<button name='task_review' class='btn btn-outline-secondary text-truncate' type='submit'>
                              <i class='fas fa-calendar-time'></i>
                              <span>" . __('Demander l\'évaluation') . "</span>
                           </button>";

            echo "</td>";
        }
        echo "</table>";

        echo "<script>";

        $queryString = http_build_query([
            "itemtype" => TicketTask::class,
            "items_id" => $this->fields["id"],
        ]);

        echo "
                 $(document).ready(function(e){
                     $('#dropdown_tasktemplates_id" . $rand . "').change(function(){
                        let value = $(this).val();
                        $.ajax({
                       url: '/ajax/task.php',
                       type: 'POST',
                       data: {
                          tasktemplates_id: value,
                          items_id: $('#dropdown_tickets_id" . $rand . "').val(),
                          itemtype: 'Ticket'
                       }
                    }).done(function (data) {
                        if (data.content !== undefined) {
                            // set textarea content
                            setRichTextEditorContent('" . $content_id . "', data.content);
                        }
        
                        if (data.taskcategories_id !== undefined) {
                            // set category
                            const taskcategories_id = isNaN(parseInt(data.taskcategories_id))
                                ? 0
                                : parseInt(data.taskcategories_id);
        
                             //need to create new DOM option, because SELECT is remotely-sourced (AJAX)
                             //see : https://select2.org/programmatic-control/add-select-clear-items#preselecting-options-in-an-remotely-sourced-ajax-select2
                             var newOption = new Option(data.taskcategories_name, taskcategories_id, true, true);
                             $('#dropdown_taskcategories_id" . $rand . "').append(newOption).trigger('change');
                        }

                        if (data.is_private !== undefined) {
                            // set is_private
                            $('#is_private_" . $rand . "').prop('checked', data.is_private == '0'? false: true);
                            }

                            if (data.state !== undefined) {
                                // Set state
                                $('#dropdown_state" . $rand . "').trigger('setValue', data.state);
                            }
                            
                            if (data.actiontime !== undefined) {
                                // Set duration
                                $('#dropdown_actiontime" . $rand . "').trigger('setValue', data.actiontime);
                            }
                            
                            if (data.users_id_tech !== undefined) {
                                // Set user
                                $('#dropdown_users_id_tech" . $rand . "').trigger('setValue', data.users_id_tech);
                            }
                            
                            if (data.groups_id_tech !== undefined) {
                                // Set group
                                $('#dropdown_groups_id_tech" . $rand . "').trigger('setValue', data.groups_id_tech);
                            }
                        });
                    });


                    const observer = new MutationObserver((mutations) => {
        
        $('#add_tickettask_plannification').off('click').click(function(e){
                        e.preventDefault();
                        
                        var modalId = glpi_ajax_dialog({
                            dialogclass: 'modal-lg',
                            bs_focus: false,
                            url: '/marketplace/dlteams/ajax/tickettask_plannif.php?" . $queryString . "',
                            title: i18n.textdomain('dlteams').__('Ajouter une planification', 'dlteams'),
                            close: function () {
            
                            },
                            fail: function () {
                                // displayAjaxMessageAfterRedirect();
                            }
                        });
                    });
                    
                    
                    $('#makecontrol').off('click').click(function(e){
                        e.preventDefault();
                        
                        var modalId = glpi_ajax_dialog({
                            dialogclass: 'modal-md',
                            bs_focus: false,
                            url: '/marketplace/dlteams/ajax/task_control.form.php',
                            params: {
                                items_id: '" . $this->fields["id"] . "'
                            },
                            title: 'Contrôle de la tâche " . $this->fields["id"] . "',
                            close: function () {
                    
                            },
                            fail: function () {
                                // displayAjaxMessageAfterRedirect();
                            }
                        });
                    });
                    
                    
                    $('#add_report').off('click').click(function(e){
                        e.preventDefault();
                        
                        var modalId = glpi_ajax_dialog({
                            dialogclass: 'modal-md',
                            bs_focus: false,
                            url: '/marketplace/dlteams/ajax/task_report.form.php',
                            params: {
                                items_id: '" . $this->fields["id"] . "'
                            },
                            title: 'Compte rendu de la tâche " . $this->fields["id"] . "',
                            close: function () {
                    
                            },
                            fail: function () {
                                // displayAjaxMessageAfterRedirect();
                            }
                        });
                    });
                    
                    
             $('#subtaskplanplanif').off('click').click(function(e){
                       e.preventDefault();
                                              
                                    $('#subtaskviewplanPlanif').load('/ajax/planning.php', {
                                        action: 'add_event_classic_form',
                                        form: 'followups',
                                        entity: " . Session::getActiveEntity() . ",
                                        itemtype: '" . TicketTask::class . "',
                                        items_id: '" . $this->fields["id"] . "',
                                        begin: $('input[name=date]').val(),
                                        duration: $('select[name=estimate_duration]').val(),
                                    });
                                    $(this).hide();
                   });
        });
        
        // Configuration de l'observateur : surveiller l'ajout et la suppression d'éléments, ainsi que les changements d'attribut
        const config = {
            childList: true,
            attributes: true,
            subtree: true // Observer les mutations sur des descendants également
        };
        
        // Commencer l'observation sur l'élément body pour couvrir tout le DOM
        observer.observe(document.body, config);
                    
                     $('#planplanif').click(function(e){
                       e.preventDefault();
                                              
                                    $('#viewplanPlanif').load('/ajax/planning.php', {
                                        action: 'add_event_classic_form',
                                        form: 'followups',
                                        entity: " . Session::getActiveEntity() . ",
                                        itemtype: '" . TicketTask::class . "',
                                        items_id: '" . $this->fields["id"] . "',
                                        begin: $('input[name=date]').val(),
                                        duration: $('select[name=estimate_duration]').val(),
                                    });
                                    $(this).hide();
                   });
                });
            ";
        echo "</script>";


        echo "<style>
            #page .tab_cadre_fixe .select2-container .select2-selection.select2-selection--single, .qtip .tab_cadre_fixe .select2-container .select2-selection.select2-selection--single, .modal .modal-body .tab_cadre_fixe .select2-container .select2-selection.select2-selection--single {
                max-width: 100%;
            }
        </style>";

        // Ajout du bandeau "créé par" et "modifié par"
        echo "<div>";
        echo "<span style='margin-top: 10px; text-align: start '>";
        $user = new User();
        $user->getFromDB($this->fields["users_id"]);
        if (isset($user->fields["firstname"]) && $user->fields["realname"])
            echo sprintf("Créé le %s par %s %s | ", date('d-m-Y H:i', strtotime($this->fields["date_creation"])), $user->fields["firstname"], $user->fields["realname"]);

        // $user = new User();
        $user->getFromDB($this->fields["users_id_editor"]);
        if (isset($user->fields["firstname"]) && $user->fields["realname"])
            echo sprintf("Mis à jour le %s par %s %s", date('d-m-Y H:i', strtotime($this->fields["date_mod"])), $user->fields["firstname"], $user->fields["realname"]);
        echo "</span>";

        if ($can_show_control_end_task)
            echo sprintf('- <span><b> Contrôle: "1 attendre fin de tâche"</b></span>');

        if ($can_make_task_control)
            echo sprintf('- <span><b> Contrôle: "1 à effectuer"</b></span>');

        $validation_state = "";
        $validation_comment = "";

        if ($tickettask_validation->getFromDBByCrit([
            "tickettasks_id" => $this->fields["id"],
            "users_id_validate" => 0
        ])) {
            if ($tickettask_validation->fields["status"] == CommonITILValidation::ACCEPTED)
                $validation_state = "Contrôle: 1 validé";

            if ($tickettask_validation->fields["status"] == CommonITILValidation::REFUSED)
                $validation_state = "Contrôle: 1 refusé";

            $validation_comment = htmlspecialchars_decode($tickettask_validation->fields["comment_submission"] ?? "");
        }


        if ($validation_state > 0)
            echo sprintf(' - <span><b style="cursor: pointer" > %s </b></span>', $validation_state);

        echo "</div>";


//        $options["addbuttons"] = [
//            'add_planification' => 'Ajouter une planification'
//        ];
        $this->showFormButtons($options);


        return true;

    }


    public function convertirTemps($tempsEnSecondes)
    {
        if ($tempsEnSecondes && is_integer($tempsEnSecondes)) {
            $heures = floor($tempsEnSecondes / 3600);
            $minutes = floor(($tempsEnSecondes % 3600) / 60);
            $secondes = $tempsEnSecondes % 60;
            return sprintf("%02d:%02d", $heures, $minutes);
        }
        return "";
    }

    public static function convertirTimestampToHour($tempsEnSecondes)
    {
        if ($tempsEnSecondes && is_integer($tempsEnSecondes)) {
            $heures = floor($tempsEnSecondes / 3600);
            $minutes = floor(($tempsEnSecondes % 3600) / 60);
            $secondes = $tempsEnSecondes % 60;
            return sprintf("%02d:%02d", $heures, $minutes);
        }
        return "";
    }

    public function defineTabs($options = [])
    {
        $ong = [];
        $ong = array();
        $this->addDefaultFormTab($ong);
        $this->addStandardTab(PluginDlteamsTicketTask_Planning::class, $ong, $options);
        if ($this->fields["tickettasks_id"] === NULL) {
            $this->addStandardTab(PluginDlteamsTicketTask_Validation::class, $ong, $options);
            $this->addStandardTab(PluginDlteamsTicketTask_Validation::class, $ong, $options);
            $this->addStandardTab(PluginDlteamsMessage::class, $ong, $options);
            $this->addStandardTab(PluginDlteamsReccurent_Task::class, $ong, $options);

            $this->addStandardTab('PluginDlteamsObject_document', $ong, $options);

            $this->addStandardTab('Notepad', $ong, $options);

        }

        $this->addStandardTab('Log', $ong, $options);

        return $ong;
    }

//    public static function addWhere($link = "", $nott = "", $itemtype = "", $ID = null, $searchtype = "", $val = "", $meta = 0)
//    {
//
//
//            $entity_filter = " AND `glpi_tickets`.`entities_id` = " . Session::getActiveEntity();
//
//        $where = "";
//        if($ID == 12){
//            $where.= " `".TicketTask::getTable()."`.`state` = ".$val.$entity_filter;
//        }
//        return $where;
//    }


    /**
     * Get default values to search engine to override
     **/
    public static function getDefaultSearchRequest()
    {
        $search = [
            'criteria' => [
                /*0 => [
                    'field' => 12,
                    'searchtype' => 'equals',
                    'value' => Planning::TODO
                ],*/
                /*1 => [
                    'field' => 9,
                    'searchtype' => 'contains',
                    'value' => ''
                ],
                2 => [
                    'field' => 10,
                    'searchtype' => 'contains',
                    'value' => ''
                ],*/
                0 => [
                    'field' => 0,
                    'searchtype' => 'contains',
                    'value' => ''
                ],
                /*4 => [
                    'field' => 16,
                    'searchtype' => 'equals',
                    'value' => '0'
                ],*/
            ],
            'sort' => 14,
            'order' => 'DESC'
        ];

        if (Session::haveRight(self::$rightname, Ticket::READALL)) {
            $search['criteria'][0]['value'] = 'notold';
        }
        return $search;
    }


    function rawSearchOptions()
    {
        $tab = [];

//        $tab = parent::rawSearchOptions();

        $tab[] = [
            'id' => 'common',
            'name' => __("Characteristics")
        ];

        $tab[] = [
            'id' => '1',
            'table' => static::getTable(),
            'field' => 'id',
            'datatype' => 'itemlink',
            'itemlink_type' => PluginDlteamsTicketTask::class,
            'name' => __("ID"),
            'massiveaction' => false,
            'searchtype' => 'contains',
        ];

        $tab[] = [
            'id' => '3',
            'table' => static::getTable(),
            'field' => 'content',
            'datatype' => 'text',
            'htmltext' => true,
            // 'toview' => true,
            // 'linkfield' => 'id',
            // 'datatype' => 'itemlink',
            'name' => __("Contenu"),
            'forcegroupby' => true,
            'massiveaction' => true,
            'searchtype' => 'contains',
        ];

        $tab[] = [
            'id' => '4',
            'table' => 'glpi_tickets',
            'field' => 'name',
            'datatype' => 'itemlink',
            'name' => __("Evenement"),
            'massiveaction' => false,
        ];

        $tab[] = [
            'id' => '17',
            'table' => 'glpi_tickets',
            'field' => 'id',
            'datatype' => 'itemlink',
            'name' => __("ID ". Ticket::getTypeName()),
            'massiveaction' => false,
        ];


        $tab[] = [
            'id' => '5',
            'table' => TaskTemplate::getTable(),
            'field' => 'name',
            'datatype' => 'dropdown',
            'name' => __("Gabarit"),
            'massiveaction' => false,
        ];

        $tab[] = [
            'id' => '8',
            'table' => static::getTable(),
            'field' => 'timeline_position',
            //'datatype' => 'itemlink',
            'name' => __("Num"),
            'massiveaction' => false,
        ];

        $tab[] = [
            'id' => '9',
            'table' => User::getTable(),
            'field' => 'name',
            'datatype' => 'dropdown',
            'linkfield' => 'users_id_tech',
            'name' => __("Acteur"),
            'massiveaction' => true,
        ];

        $tab[] = [
            'id' => '10',
            'table' => Group::getTable(),
            'field' => 'name',
            'datatype' => 'dropdown',
            'linkfield' => 'groups_id_tech',
            'name' => __("Groupe"),
            'massiveaction' => true,
        ];

        $tab[] = [
            'id' => '11',
            'table' => static::getTable(),
            'field' => 'actiontime',
            'name' => __("Durée"),
            'datatype' => 'timestamp',
            'massiveaction' => true,
        ];

        $tab[] = [
            'id' => '12',
            'table' => static::getTable(),
            'field' => 'state',
            'name' => __('Status'),
            'datatype' => 'specific',
            'searchtype' => ['equals'],
            'searchequalsonfield' => true,
            'massiveaction' => true,
            'forcegroupby' => true,
        ];

        $tab[] = [
            'id' => '13',
            'table' => $this->getTable(),
            'field' => 'date_creation',
            'name' => __("Creation date"),
            'massiveaction' => false,
            'datatype' => 'text',
        ];

        $tab[] = [
            'id' => '14',
            'table' => $this->getTable(),
            'field' => 'date_mod',
            'name' => __("Last update"),
            'massiveaction' => false,
            'datatype' => 'text',
        ];

        /*Html::showDateTimeField('begin', [
                // 'value' => sprintf ("%s", date('d/m/Y H:i', strtotime($this->fields["begin"]))) != '' ? $this->fields["begin"] : '',
                'value' => date('d/m/Y H:i') != '' ? $this->fields["begin"] : '',
                // 'rand' => $rand,
                'display' => true,
                'datatype' => 'datetime',
        ]);*/

        $tab[] = [
            'id' => '15',
            'table' => $this->getTable(),
            'field' => 'begin',
            'name' => __("Planning"),
            'searchtype' => 'equals',
            'massiveaction' => false,
            'datatype' => 'datetime',
        ];

        $tab[] = [
            'id' => '20',
            'table' => $this->getTable(),
            'field' => 'date',
            'name' => __('Echéance'),
            'searchtype' => 'equals',
            'datatype' => 'datetime'
        ];

        $tab[] = [
            'id' => '16',
            'table' => PluginDlteamsTicketTask_Validation::getTable(),
            'field' => 'status',
            'name' => __('Status de validation'),
            'datatype' => 'specific',
            'searchtype' => ['equals'],
            'searchequalsonfield' => true,
            'massiveaction' => true,
            'forcegroupby' => true,
            'joinparams'    => [
                'jointype' => 'child',
                'nosort'             => true,
            ]
        ];

        $entityId = $_SESSION["glpiactive_entity"];
        return $tab;
    }

    public static function getTaskClass()
    {
        // TODO: Implement getTaskClass() method.
    }

    public static function getDefaultValues($entity = 0)
    {
        // TODO: Implement getDefaultValues() method.
    }

    public static function getItemLinkClass(): string
    {
        // TODO: Implement getItemLinkClass() method.
    }

    public static function getContentTemplatesParametersClass(): string
    {
        // TODO: Implement getContentTemplatesParametersClass() method.
    }


    public function addDefaultFormTab(array &$ong)
    {

        $timeline = $this->getTimelineItems(['with_logs' => false]);
        $nb_elements = count($timeline);
        $label = $this->getTypeName(1);
//        if ($nb_elements > 0) {
//            $label .= " <span class='badge'>$nb_elements</span>";
//        }

        $ong[$this->getType() . '$main'] = $label;
        return $this;
    }

    public function getTimelineItems(array $options = [])
    {

        $params = [
            'with_documents' => true,
            'with_logs' => true,
            'with_validations' => true,
            'sort_by_date_desc' => $_SESSION['glpitimeline_order'] == CommonITILObject::TIMELINE_ORDER_REVERSE,

            // params used by notifications process (as session cannot be used there)
            'check_view_rights' => true,
            'hide_private_items' => false,
        ];

        if (array_key_exists('bypass_rights', $options) && $options['bypass_rights']) {
            Toolbox::deprecated('Using `bypass_rights` parameter is deprecated.');
            $params['check_view_rights'] = false;
        }
        if (array_key_exists('expose_private', $options) && $options['expose_private']) {
            Toolbox::deprecated('Using `expose_private` parameter is deprecated.');
            $params['hide_private_items'] = false;
        }
        if (array_key_exists('is_self_service', $options) && $options['is_self_service']) {
            Toolbox::deprecated('Using `is_self_service` parameter is deprecated.');
            $params['hide_private_items'] = false;
        }

        if (is_array($options) && count($options)) {
            foreach ($options as $key => $val) {
                $params[$key] = $val;
            }
        }

        if ($this->isNewItem()) {
            return [];
        }

        if ($params['check_view_rights'] && !$this->canViewItem()) {
            return [];
        }

        $objType = static::getType();
        $foreignKey = static::getForeignKeyField();
        $timeline = [];

        $canupdate_parent = $this->canUpdateItem() && !in_array($this->fields['status'], $this->getClosedStatusArray());

        //checks rights
        $restrict_fup = $restrict_task = [];
        if (
            $params['hide_private_items']
            || ($params['check_view_rights'] && !Session::haveRight("followup", ITILFollowup::SEEPRIVATE))
        ) {
            if (!$params['check_view_rights']) {
                // notification case, we cannot rely on session
                $restrict_fup = [
                    'is_private' => 0,
                ];
            } else {
                $restrict_fup = [
                    'OR' => [
                        'is_private' => 0,
                        'users_id' => Session::getCurrentInterface() === "central" ? (int)Session::getLoginUserID() : 0,
                    ]
                ];
            }
        }

        $restrict_fup['itemtype'] = static::getType();
        $restrict_fup['items_id'] = $this->getID();

        $taskClass = $objType . "Task";
        $task_obj = new TicketTask();
        if (
            $task_obj->maybePrivate()
            && (
                $params['hide_private_items']
                || ($params['check_view_rights'] && !Session::haveRight($task_obj::$rightname, CommonITILTask::SEEPRIVATE))
            )
        ) {
            if (!$params['check_view_rights']) {
                // notification case, we cannot rely on session
                $restrict_task = [
                    'is_private' => 0,
                ];
            } else {
                $restrict_task = [
                    'OR' => [
                        'is_private' => 0,
                        'users_id' => Session::getCurrentInterface() === "central" ? (int)Session::getLoginUserID() : 0,
                    ]
                ];
            }
        }

        // Add followups to timeline
        $followup_obj = new ITILFollowup();
        if (!$params['check_view_rights'] || $followup_obj->canview()) {
            $followups = $followup_obj->find(
                ['items_id' => $this->getID()] + $restrict_fup,
                ['date_creation DESC', 'id DESC']
            );

            foreach ($followups as $followups_id => $followup_row) {
                // Safer to use a clean object to load our data
                $followup = new ITILFollowup();
                $followup->setParentItem($this);
                $followup->fields = $followup_row;
                $followup->post_getFromDB();

                if (!$params['check_view_rights'] || $followup->canViewItem()) {
                    $followup_row['can_edit'] = $followup->canUpdateItem();
                    $followup_row['can_promote'] =
                        Session::getCurrentInterface() === 'central'
                        && $this instanceof Ticket
                        && Ticket::canCreate();
                    $timeline["ITILFollowup_" . $followups_id] = [
                        'type' => ITILFollowup::class,
                        'item' => $followup_row,
                        'object' => $followup,
                        'itiltype' => 'Followup'
                    ];
                }
            }
        }

        // Add tasks to timeline
        if (!$params['check_view_rights'] || $task_obj->canview()) {
            $tasks = $task_obj->find(
                [$foreignKey => $this->getID()] + $restrict_task,
                'date_creation DESC'
            );

            foreach ($tasks as $tasks_id => $task_row) {
                // Safer to use a clean object to load our data
//                $task = new $taskClass();
                $task = new self(); // dlteams
                $task->fields = $task_row;
                $task->post_getFromDB();

                if (!$params['check_view_rights'] || $task->canViewItem()) {
                    $task_row['can_edit'] = $task->canUpdateItem();
                    $task_row['can_promote'] =
                        Session::getCurrentInterface() === 'central'
                        && $this instanceof Ticket
                        && Ticket::canCreate();
                    $timeline[$task::getType() . "_" . $tasks_id] = [
                        'type' => $taskClass,
                        'item' => $task_row,
                        'object' => $task,
                        'itiltype' => 'Task'
                    ];
                }
            }
        }

        // Add solutions to timeline
        $solution_obj = new ITILSolution();
        $solution_items = $solution_obj->find([
            'itemtype' => static::getType(),
            'items_id' => $this->getID()
        ]);

        foreach ($solution_items as $solution_item) {
            // Safer to use a clean object to load our data
            $solution = new ITILSolution();
            $solution->setParentItem($this);
            $solution->fields = $solution_item;
            $solution->post_getFromDB();

            $timeline["ITILSolution_" . $solution_item['id']] = [
                'type' => ITILSolution::class,
                'itiltype' => 'Solution',
                'item' => [
                    'id' => $solution_item['id'],
                    'content' => $solution_item['content'],
                    'date' => $solution_item['date_creation'],
                    'users_id' => $solution_item['users_id'],
                    'solutiontypes_id' => $solution_item['solutiontypes_id'],
                    'can_edit' => $objType::canUpdate() && $this->canSolve(),
                    'timeline_position' => self::TIMELINE_RIGHT,
                    'users_id_editor' => $solution_item['users_id_editor'],
                    'date_creation' => $solution_item['date_creation'],
                    'date_mod' => $solution_item['date_mod'],
                    'users_id_approval' => $solution_item['users_id_approval'],
                    'date_approval' => $solution_item['date_approval'],
                    'status' => $solution_item['status']
                ],
                'object' => $solution,
            ];
        }

        // Add validation to timeline
        $validation_class = $objType . "Validation";
        if (
            class_exists($validation_class) && $params['with_validations']
            && (!$params['check_view_rights'] || $validation_class::canView())
        ) {
            $valitation_obj = new $validation_class();
            $validations = $valitation_obj->find([
                $foreignKey => $this->getID()
            ]);

            foreach ($validations as $validations_id => $validation_row) {
                // Safer to use a clean object to load our data
                $validation = new $validation_class();
                $validation->fields = $validation_row;
                $validation->post_getFromDB();

                $canedit = $valitation_obj->can($validations_id, UPDATE);
                $cananswer = ($validation_row['users_id_validate'] === Session::getLoginUserID() &&
                    $validation_row['status'] == CommonITILValidation::WAITING);
                $user = new User();
                $user->getFromDB($validation_row['users_id_validate']);

                $request_key = $valitation_obj::getType() . '_' . $validations_id
                    . (empty($validation_row['validation_date']) ? '' : '_request'); // If no answer, no suffix to see attached documents on request
                $timeline[$request_key] = [
                    'type' => $validation_class,
                    'item' => [
                        'id' => $validations_id,
                        'date' => $validation_row['submission_date'],
                        'content' => __('Validation request') . " <i class='ti ti-arrow-right'></i><i class='ti ti-user text-muted me-1'></i>" . $user->getlink(),
                        'comment_submission' => $validation_row['comment_submission'],
                        'users_id' => $validation_row['users_id'],
                        'can_edit' => $canedit,
                        'can_answer' => $cananswer,
                        'users_id_validate' => $validation_row['users_id_validate'],
                        'timeline_position' => $validation_row['timeline_position']
                    ],
                    'itiltype' => 'Validation',
                    'class' => 'validation-request ' .
                        ($validation_row['status'] == CommonITILValidation::WAITING ? "validation-waiting" : "") .
                        ($validation_row['status'] == CommonITILValidation::ACCEPTED ? "validation-accepted" : "") .
                        ($validation_row['status'] == CommonITILValidation::REFUSED ? "validation-refused" : ""),
                    'item_action' => 'validation-request',
                    'object' => $validation,
                ];

                if (!empty($validation_row['validation_date'])) {
                    $timeline[$valitation_obj::getType() . "_" . $validations_id] = [
                        'type' => $validation_class,
                        'item' => [
                            'id' => $validations_id,
                            'date' => $validation_row['validation_date'],
                            'content' => __('Validation request answer') . " : " .
                                _sx('status', ucfirst($validation_class::getStatus($validation_row['status']))),
                            'comment_validation' => $validation_row['comment_validation'],
                            'users_id' => $validation_row['users_id_validate'],
                            'status' => "status_" . $validation_row['status'],
                            'can_edit' => $validation_row['users_id_validate'] === Session::getLoginUserID(),
                            'timeline_position' => $validation_row['timeline_position'],
                        ],
                        'class' => 'validation-answer',
                        'itiltype' => 'Validation',
                        'item_action' => 'validation-answer',
                        'object' => $validation,
                    ];
                }
            }
        }

        // Add documents to timeline
        if ($params['with_documents']) {
            $document_item_obj = new Document_Item();
            $document_obj = new Document();
            $document_items = $document_item_obj->find([
                $this->getAssociatedDocumentsCriteria(!$params['check_view_rights']),
                'timeline_position' => ['>', self::NO_TIMELINE]
            ]);
            foreach ($document_items as $document_item) {
                if (!$document_obj->getFromDB($document_item['documents_id'])) {
                    // Orphan `Document_Item`
                    continue;
                }

                $item = $document_obj->fields;
                $item['date'] = $document_item['date'] ?? $document_item['date_creation'];
                // #1476 - set date_creation, date_mod and owner to attachment ones
                $item['date_creation'] = $document_item['date_creation'];
                $item['date_mod'] = $document_item['date_mod'];
                $item['users_id'] = $document_item['users_id'];
                $item['documents_item_id'] = $document_item['id'];

                $item['timeline_position'] = $document_item['timeline_position'];
                $item['_can_edit'] = Document::canUpdate() && $document_obj->canUpdateItem();
                $item['_can_delete'] = Document::canDelete() && $document_obj->canDeleteItem() && $canupdate_parent;

                $timeline_key = $document_item['itemtype'] . "_" . $document_item['items_id'];
                if ($document_item['itemtype'] == static::getType()) {
                    // document associated directly to itilobject
                    $timeline["Document_" . $document_item['documents_id']] = [
                        'type' => 'Document_Item',
                        'item' => $item,
                        'object' => $document_obj,
                    ];
                } elseif (isset($timeline[$timeline_key])) {
                    // document associated to a sub item of itilobject
                    if (!isset($timeline[$timeline_key]['documents'])) {
                        $timeline[$timeline_key]['documents'] = [];
                    }

                    $docpath = GLPI_DOC_DIR . "/" . $item['filepath'];
                    $is_image = Document::isImage($docpath);
                    $sub_document = [
                        'type' => 'Document_Item',
                        'item' => $item,
                    ];
                    if ($is_image) {
                        $sub_document['_is_image'] = true;
                        $sub_document['_size'] = getimagesize($docpath);
                    }
                    $timeline[$timeline_key]['documents'][] = $sub_document;
                }
            }
        }

        // Add logs to timeline
        if ($params['with_logs'] && Session::getCurrentInterface() == "central") {
            //add logs to timeline
            $log_items = Log::getHistoryData($this, 0, 0, [
                'OR' => [
                    'id_search_option' => ['>', 0],
                    'itemtype_link' => ['User', 'Group', 'Supplier'],
                ]
            ]);

            foreach ($log_items as $log_row) {
                // Safer to use a clean object to load our data
                $log = new Log();
                $log->fields = $log_row;
                $log->post_getFromDB();

                $content = $log_row['change'];
                if (strlen($log_row['field']) > 0) {
                    $content = sprintf(__("%s: %s"), $log_row['field'], $content);
                }
                $content = "<i class='fas fa-history me-1' title='" . __("Log entry") . "' data-bs-toggle='tooltip'></i>" . $content;
                $timeline["Log_" . $log_row['id']] = [
                    'type' => 'Log',
                    'class' => 'text-muted d-none',
                    'item' => [
                        'id' => $log_row['id'],
                        'content' => $content,
                        'date' => $log_row['date_mod'],
                        'users_id' => 0,
                        'can_edit' => false,
                        'timeline_position' => self::TIMELINE_LEFT,
                    ],
                    'object' => $log,
                ];
            }
        }
        Plugin::doHook(\Glpi\Plugin\Hooks::SHOW_IN_TIMELINE, ['item' => $this, 'timeline' => &$timeline]);

        //sort timeline items by date. If items have the same date, sort by id
        $reverse = $params['sort_by_date_desc'];
        usort($timeline, function ($a, $b) use ($reverse) {
            $date_a = $a['item']['date_creation'] ?? $a['item']['date'];
            $date_b = $b['item']['date_creation'] ?? $b['item']['date'];
            if($date_a && $date_b)
                $diff = strtotime($date_a) - strtotime($date_b);
            else $diff = 0;
            if ($diff === 0) {
                $diff = $a['item']['id'] - $b['item']['id'];
            }
            return $reverse ? 0 - $diff : $diff;
        });

        return $timeline;
    }

    public function post_addItem()
    {
        parent::post_addItem(); // TODO: Change the autogenerated stub
    }


    /**
     * Returns criteria that can be used to get documents related to current instance.
     *
     * @return array
     */
    public function getAssociatedDocumentsCriteria($bypass_rights = false): array
    {
//        $task_class = $this->getType() . 'Task';
        $task_class = TicketTask::class;
        /** @var DBMysql $DB */
        global $DB; // Used to get subquery results - better performance

        $or_crits = [
            // documents associated to ITIL item directly
            [
                Document_Item::getTableField('itemtype') => $this->getType(),
                Document_Item::getTableField('items_id') => $this->getID(),
            ],
        ];

        // documents associated to followups
        if ($bypass_rights || ITILFollowup::canView()) {
            $fup_crits = [
                ITILFollowup::getTableField('itemtype') => $this->getType(),
                ITILFollowup::getTableField('items_id') => $this->getID(),
            ];
            if (!$bypass_rights && !Session::haveRight(ITILFollowup::$rightname, ITILFollowup::SEEPRIVATE)) {
                $fup_crits[] = [
                    'OR' => ['is_private' => 0, 'users_id' => Session::getLoginUserID()],
                ];
            }
            // Run the subquery separately. It's better for huge databases
            $iterator_tmp = $DB->request([
                'SELECT' => 'id',
                'FROM' => ITILFollowup::getTable(),
                'WHERE' => $fup_crits,
            ]);
            $arr_values = array_column(iterator_to_array($iterator_tmp, false), 'id');
            if (count($arr_values) > 0) {
                $or_crits[] = [
                    Document_Item::getTableField('itemtype') => ITILFollowup::getType(),
                    Document_Item::getTableField('items_id') => $arr_values,
                ];
            }
        }

        // documents associated to solutions
        if ($bypass_rights || ITILSolution::canView()) {
            // Run the subquery separately. It's better for huge databases
            $iterator_tmp = $DB->request([
                'SELECT' => 'id',
                'FROM' => ITILSolution::getTable(),
                'WHERE' => [
                    ITILSolution::getTableField('itemtype') => $this->getType(),
                    ITILSolution::getTableField('items_id') => $this->getID(),
                ],
            ]);
            $arr_values = array_column(iterator_to_array($iterator_tmp, false), 'id');
            if (count($arr_values) > 0) {
                $or_crits[] = [
                    Document_Item::getTableField('itemtype') => ITILSolution::getType(),
                    Document_Item::getTableField('items_id') => $arr_values,
                ];
            }
        }

        // documents associated to ticketvalidation
        $validation_class = static::getType() . 'Validation';
        if (class_exists($validation_class) && ($bypass_rights || $validation_class::canView())) {
            // Run the subquery separately. It's better for huge databases
            $iterator_tmp = $DB->request([
                'SELECT' => 'id',
                'FROM' => $validation_class::getTable(),
                'WHERE' => [
                    $validation_class::getTableField($validation_class::$items_id) => $this->getID(),
                ],
            ]);
            $arr_values = array_column(iterator_to_array($iterator_tmp, false), 'id');
            if (count($arr_values) > 0) {
                $or_crits[] = [
                    Document_Item::getTableField('itemtype') => $validation_class::getType(),
                    Document_Item::getTableField('items_id') => $arr_values,
                ];
            }
        }

        // documents associated to tasks
        if ($bypass_rights || $task_class::canView()) {
            $tasks_crit = [
                $this->getForeignKeyField() => $this->getID(),
            ];
            if (!$bypass_rights && !Session::haveRight($task_class::$rightname, CommonITILTask::SEEPRIVATE)) {
                $tasks_crit[] = [
                    'OR' => ['is_private' => 0, 'users_id' => Session::getLoginUserID()],
                ];
            }
            // Run the subquery separately. It's better for huge databases
            $iterator_tmp = $DB->request([
                'SELECT' => 'id',
                'FROM' => $task_class::getTable(),
                'WHERE' => $tasks_crit,
            ]);
            $arr_values = array_column(iterator_to_array($iterator_tmp, false), 'id');
            if (count($arr_values) > 0) {
                $or_crits[] = [
                    'glpi_documents_items.itemtype' => $task_class::getType(),
                    'glpi_documents_items.items_id' => $arr_values,
                ];
            }
        }
        return ['OR' => $or_crits];
    }
}
