<?php

/**
 * ---------------------------------------------------------------------
 *
 * GLPI - Gestionnaire Libre de Parc Informatique
 *
 * http://glpi-project.org
 *
 * @copyright 2015-2025 Teclib' and contributors.
 * @copyright 2003-2014 by the INDEPNET Development Team.
 * @licence   https://www.gnu.org/licenses/gpl-3.0.html
 *
 * ---------------------------------------------------------------------
 *
 * LICENSE
 *
 * This file is part of GLPI.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * ---------------------------------------------------------------------
 */

/**
 * Notepad class
 *
 * @since 0.85
 **/
class Notepad extends CommonDBChild
{
    // From CommonDBChild
    public static $itemtype        = 'itemtype';
    public static $items_id        = 'items_id';
    public $dohistory              = false;
    public $auto_message_on_action = false; // Link in message can't work'
    public static $logs_for_parent = true;


    public static function getTypeName($nb = 0)
    {
        //TRANS: Always plural
        return _n('Note, lien', 'Notes, liens', $nb);
    }

//    fork dlteams
    public static function getIcon()
    {
        return 'fas fa-sticky-note';
    }

    public function getLogTypeID()
    {
        return [$this->fields['itemtype'], $this->fields['items_id']];
    }

    public static function showCentralList($start = 0, $status = 'process', $showgrouptickets = true)
    {
        global $DB;

//        if (!Session::haveRight('plugin_dlteams_protectivemeasure', READ)) {
//            return false;
//        }

        $iterator = $DB->request([
            'FROM' => self::getTable(),
            'WHERE' => [
                'entities_id' => Session::getActiveEntity()
            ],
            'ORDER' => ['date_mod DESC'],
            'LIMIT' => $_SESSION['glpidisplay_count_on_home'] ?? 10
        ]);

        $total_row_count = count($iterator);
        $displayed_row_count = min((int)$_SESSION['glpidisplay_count_on_home'], $total_row_count);

        if ($total_row_count > 0) {
            // Titre principal avec lien
            $main_header = "<a href=\"" . self::getSearchURL() . "\">" .
                Html::makeTitle(__('Recent Notes, liens', 'dlteams'), $displayed_row_count, $total_row_count) . "</a>";

            echo "<div class='table-responsive'>";
            echo "<table class='table table-borderless table-striped table-hover card-table'>";
            echo "<thead>";
            echo "<tr>";
            echo "<th colspan='3'>" . $main_header . "</th>";
            echo "</tr>";
            echo "<tr>";
            echo "<th>" . __('ID') . "</th>";
            echo "<th>" . __('Name') . "</th>";
            echo "<th>" . __('Date modification') . "</th>";
            echo "</tr>";
            echo "</thead>";
            echo "<tbody>";

            foreach ($iterator as $data) {
                $url = self::getFormURLWithID($data['id']);
                echo "<tr>";
                echo "<td><a href='" . $url . "' class='btn btn-sm btn-outline-primary'>" . $data['id'] . "</a></td>";
                echo "<td>" . htmlspecialchars($data['name']) . "</td>";
                echo "<td class='text-muted'>" . Html::convDateTime($data['date_mod']) . "</td>";
                echo "</tr>";
            }

            echo "</tbody>";
            echo "</table>";
            echo "</div>";

            if ($total_row_count > $displayed_row_count) {
                echo "<div class='text-center mt-2'>";
                echo "<a href='" . self::getSearchURL() . "' class='btn btn-sm btn-outline-secondary'>";
                echo sprintf(__('Show all %d items'), $total_row_count);
                echo "</a>";
                echo "</div>";
            }
        } else {
            echo "<div class='text-center text-muted p-4'>";
            echo "<i class='fas fa-inbox fa-2x mb-2'></i><br>";
            echo __('No item found');
            echo "</div>";
        }
    }


    public function canCreateItem()
    {

        if (
            isset($this->fields['itemtype'])
            && ($item = getItemForItemtype($this->fields['itemtype']))
        ) {
            return Session::haveRight($item::$rightname, UPDATENOTE);
        }
        return false;
    }


    public function canUpdateItem()
    {

        if (
            isset($this->fields['itemtype'])
            && ($item = getItemForItemtype($this->fields['itemtype']))
        ) {
            return Session::haveRight($item::$rightname, UPDATENOTE);
        }
        return false;
    }


    public function prepareInputForAdd($input)
    {

        $input['users_id']             = Session::getLoginUserID();
        $input['users_id_lastupdater'] = Session::getLoginUserID();
        $input['entities_id'] = Session::getActiveEntity(); // fork dlteams
        return $input;
    }


    public function prepareInputForUpdate($input)
    {

        $input['users_id_lastupdater'] = Session::getLoginUserID();
        return $input;
    }

    public function getTabNameForItem(CommonGLPI $item, $withtemplate = 0)
    {

//        if (Session::haveRight($item::$rightname, READNOTE)) {
        $nb = 0;
        if ($_SESSION['glpishow_count_on_tabs']) {
            $nb = self::countForItem($item);
        }

//            fork dlteams
        $nb += countElementsInTable(
            'glpi_manuallinks',
            [
                'itemtype'  => $item->getType(),
                'items_id'  => $item->fields[$item->getIndexName()],
            ]
        );

        return self::createTabEntry(self::getTypeName(Session::getPluralNumber()), $nb);
//        }
        return '';
    }


    /**
     * @param $item            CommonGLPI object
     * @param $tabnum          (default 1)
     * @param $withtemplate    (default 0)
     **/
    public static function displayTabContentForItem(CommonGLPI $item, $tabnum = 1, $withtemplate = 0)
    {
        ManualLink::showForItem($item);
        static::showForItem($item, $withtemplate);
        return true;
    }


    /**
     * @param $item    CommonDBTM object
     *
     * @return number
     **/
    public static function countForItem(CommonDBTM $item)
    {

        return countElementsInTable(
            'glpi_notepads',
            ['itemtype' => $item->getType(),
                'items_id' => $item->getID()
            ]
        );
    }


    /**
     * @param $item   CommonDBTM object
     **/
    public static function getAllForItem(CommonDBTM $item)
    {
        /** @var \DBmysql $DB */
        global $DB;

        $data = [];
        $iterator = $DB->request([
            'SELECT'    => [
                'glpi_notepads.*',
                'glpi_users.picture'
            ],
            'FROM'      => self::getTable(),
            'LEFT JOIN' => [
                'glpi_users'   => [
                    'ON' => [
                        self::getTable()  => 'users_id_lastupdater',
                        'glpi_users'      => 'id'
                    ]
                ]
            ],
            'WHERE'     => [
                'itemtype'  => $item->getType(),
                'items_id'  => $item->getID()
            ],
            'ORDERBY'   => 'date_mod DESC'
        ]);

        foreach ($iterator as $note) {
            $data[] = $note;
        }
        return $data;
    }


    public static function rawSearchOptionsToAdd()
    {
        $tab = [];
        $name = _n('Note', 'Notes', Session::getPluralNumber());

        $tab[] = [
            'id'                 => 'notepad',
            'name'               => $name
        ];

        $tab[] = [
            'id'                 => '200',
            'table'              => 'glpi_notepads',
            'field'              => 'content',
            'name'               => $name,
            'datatype'           => 'text',
            'joinparams'         => [
                'jointype'           => 'itemtype_item'
            ],
            'forcegroupby'       => true,
            'splititems'         => true,
            'massiveaction'      => false
        ];

        $tab[] = [
            'id'                 => '201',
            'table'              => 'glpi_notepads',
            'field'              => 'date_creation',
            'name'               => __('Creation date'),
            'datatype'           => 'datetime',
            'joinparams'         => [
                'jointype'           => 'itemtype_item'
            ],
            'forcegroupby'       => true,
            'massiveaction'      => false
        ];

        $tab[] = [
            'id'                 => '202',
            'table'              => 'glpi_users',
            'field'              => 'name',
            'name'               => __('Writer'),
            'datatype'           => 'dropdown',
            'forcegroupby'       => true,
            'massiveaction'      => false,
            'joinparams'         => [
                'beforejoin'         => [
                    'table'              => 'glpi_notepads',
                    'joinparams'         => [
                        'jointype'           => 'itemtype_item'
                    ]
                ]
            ]
        ];

        $tab[] = [
            'id'                 => '203',
            'table'              => 'glpi_notepads',
            'field'              => 'date_mod',
            'name'               => __('Last update'),
            'datatype'           => 'datetime',
            'joinparams'         => [
                'jointype'           => 'itemtype_item'
            ],
            'forcegroupby'       => true,
            'massiveaction'      => false
        ];

        $tab[] = [
            'id'                 => '204',
            'table'              => 'glpi_users',
            'field'              => 'name',
            'linkfield'          => 'users_id_lastupdater',
            'name'               => __('Last updater'),
            'datatype'           => 'dropdown',
            'forcegroupby'       => true,
            'massiveaction'      => false,
            'joinparams'         => [
                'beforejoin'         => [
                    'table'              => 'glpi_notepads',
                    'joinparams'         => [
                        'jointype'           => 'itemtype_item'
                    ]
                ]
            ]
        ];

        return $tab;
    }

    function rawSearchOptions() {

        $tab = [];

        $tab[] = [
            'id'                 => 'common',
            'name'               => __("Characteristics")
        ];

        $tab[] = [
            'id'                 => '9', // fork dlteams
            'table'              => $this->getTable(),
            'field'              => 'itemtype',
            'name'               => __("Type"),
            'datatype'           => 'itemtypename',
            'massiveaction'      => true, // fork dlteams
            'autocomplete'       => true,
        ];

        $tab[] = [
            'id'                 => '2',
            'table'              => $this->getTable(),
            'field'              => 'id',
            'name'               => __("ID"),
            'massiveaction'      => false,
            'datatype'           => 'itemlink',
        ];

//      $tab[] = [
//         'id'                 => '3',
//         'table'              => 'glpi_plugin_dlteams_rightmeasurecategories',
//         'field'              => 'name',
//         'name'               => __("Type", 'dlteams'),
//         'searchtype'         => 'equals',
//         'massiveaction'      => true,
//         'datatype'           => 'specific'
//      ];

        $tab[] = [
            'id'                 => '4',
            'table'              => $this->getTable(),
            'field'              => 'content',
            'name'               => __("Content"),
            'datatype'           => 'text',
            'htmltext'           => true,
            'toview'             => true,
            'massiveaction'      => true,
        ];


        $tab[] = [
            'id'                 => '6',
            'table'              => 'glpi_entities',
            'field'              => 'completename',
            'name'               => __("Entity"),
            'massiveaction'      => true,
            'datatype'           => 'dropdown',
        ];

        $tab[] = [
            'id'                 => '11', // fork dlteams
            'table'              => $this->getTable(),
            'field'              => 'date_mod',
            'name'               => __("Date modification"),
            'datatype'           => 'date',
            'massiveaction'      => true, // fork dlteams
            'autocomplete'       => true,
        ];

        $tab[] = [
            'id'                 => '12', // fork dlteams
            'table'              => $this->getTable(),
            'field'              => 'date_creation',
            'name'               => __("Date création"),
            'datatype'           => 'date',
            'massiveaction'      => true, // fork dlteams
            'autocomplete'       => true,
        ];

        $tab[] = [
            'id'                 => '13', // fork dlteams
            'table'              => $this->getTable(),
            'field'              => 'users_id',
            'name'               => __("Writer"),
            'datatype'           => 'text',
            'massiveaction'      => true, // fork dlteams
            'autocomplete'       => true,
        ];

        return $tab;
    }

    /**
     * Show notepads for an item
     *
     * @param $item                  CommonDBTM object
     * @param $withtemplate integer  template or basic item (default 0)
     **/
    public static function showForItem(CommonDBTM $item, $withtemplate = 0)
    {
        if (!Session::haveRight($item::$rightname, READNOTE)) {
            return false;
        }
        $notes   = static::getAllForItem($item);
        $rand    = mt_rand();
        $canedit = Session::haveRight($item::$rightname, UPDATENOTE);

        $showuserlink = 0;
        if (User::canView()) {
            $showuserlink = 1;
        }

        if (
            $canedit
            && !(!empty($withtemplate) && ($withtemplate == 2))
        ) {
            echo "<div class='boxnote center'>";

            echo "<div class='boxnoteleft'></div>";
            echo "<form name='addnote_form$rand' id='addnote_form$rand' ";
            echo " method='post' action='" . Toolbox::getItemTypeFormURL('Notepad') . "'>";
            echo Html::hidden('itemtype', ['value' => $item->getType()]);
            echo Html::hidden('items_id', ['value' => $item->getID()]);

            echo "<div class='boxnotecontent'>";
//            fork dlteams
//            echo "<textarea name='content' class='form-control' rows='7'></textarea>";
            Html::textarea(['name' => 'content',
                'value' =>  "" ,
                'enable_fileupload' => false,
                'enable_richtext' => true,
                'cols' => 100,
                'rows' => 5
            ]);
            echo "</div>"; // box notecontent

            echo "<div class='boxnoteright'><br>";
            echo Html::submit(_x('button', 'Add'), ['name' => 'add', 'class' => 'btn btn-primary']);
            echo "</div>";

            Html::closeForm();
            echo "</div>"; // boxnote
        }

        if (count($notes)) {
            foreach ($notes as $note) {
                $id = 'note' . $note['id'] . $rand;
                $classtoadd = '';
                if ($canedit) {
                    $classtoadd = " pointer";
                }
                echo "<div class='boxnote' id='view$id'>";

                echo "<div class='boxnoteleft'>";
                $thumbnail_url = User::getThumbnailURLForPicture($note['picture']);
                $user = new User();
                $user->getFromDB($note['users_id_lastupdater']);
                $style = !empty($thumbnail_url) ? "background-image: url('$thumbnail_url'); background-color: inherit;" : ("background-color: " . $user->getUserInitialsBgColor());
                echo '<a href="' . $user->getLinkURL() . '">';
                $user_name = formatUserName(
                    $user->getID(),
                    $user->fields['name'],
                    $user->fields['realname'],
                    $user->fields['firstname']
                );
                echo '<span class="avatar avatar-md rounded" style="' . $style . '" title="' . $user_name . '">';
                if (empty($thumbnail_url)) {
                    echo $user->getUserInitials();
                }
                echo '</span>';
                echo '</a>';
                echo "</div>"; // boxnoteleft

                echo "<div class='boxnotecontent'>";
//                fork dlteams
                $noteid = $note['id'];
                echo "<span>$noteid</span>";

                echo "<div class='boxnotefloatright'>";
                $username = NOT_AVAILABLE;
                if ($note['users_id_lastupdater']) {
                    $username = getUserName($note['users_id_lastupdater'], $showuserlink);
                }
                $update = sprintf(
                    __('Last update by %1$s on %2$s'),
                    $username,
                    Html::convDateTime($note['date_mod'])
                );
                $username = NOT_AVAILABLE;
                if ($note['users_id']) {
                    $username = getUserName($note['users_id'], $showuserlink);
                }
                $create = sprintf(
                    __('Create by %1$s on %2$s'),
                    $username,
                    Html::convDateTime($note['date_creation'])
                );
                printf(__('%1$s / %2$s'), $update, $create);
                echo "</div>"; // floatright

                echo "<div class='boxnotetext $classtoadd' ";
                if ($canedit) {
                    echo "onclick=\"" . Html::jsHide("view$id") . " " .
                        Html::jsShow("edit$id") . "\"";
                }
                echo ">";
                $content = nl2br($note['content']);
                if (empty($content)) {
                    $content = NOT_AVAILABLE;
                }
                echo htmlspecialchars_decode($content) . '</div>'; // boxnotetext // dlteams

                echo "</div>"; // boxnotecontent
                echo "<div class='boxnoteright'>";
                if ($canedit) {
                    Html::showSimpleForm(
                        Toolbox::getItemTypeFormURL('Notepad'),
                        ['purge' => 'purge'],
                        _x('button', 'Delete permanently'),
                        ['id'   => $note['id']],
                        'ti-circle-x',
                        '',
                        __('Confirm the final deletion?')
                    );
                }
                echo "</div>"; // boxnoteright
                echo "</div>"; // boxnote

                if ($canedit) {
                    echo "<div class='boxnote starthidden' id='edit$id'>";
                    echo "<form name='update_form$id$rand' id='update_form$id$rand' ";
                    echo " method='post' action='" . Toolbox::getItemTypeFormURL('Notepad') . "'>";

                    echo "<div class='boxnoteleft'></div>";
                    echo "<div class='boxnotecontent'>";
                    echo Html::hidden('id', ['value' => $note['id']]);
//                    echo "<textarea name='content' rows=5 cols=100>" . $note['content'] . "</textarea>";
                    Html::textarea(['name' => 'content',
                        'value' =>  $note['content'] ,
                        'enable_fileupload' => false,
                        'enable_richtext' => true,
                        'cols' => 100,
                        'rows' => 5
                    ]);

                    echo "</div>"; // boxnotecontent

                    echo "<div class='boxnoteright'><br>";
                    echo Html::submit(_x('button', 'Update'), ['name' => 'update']);
                    echo "</div>"; // boxnoteright

                    Html::closeForm();
                    echo "</div>"; // boxnote
                }
            }
        }

//        fork dlteams
        echo "<style>
        p {
            margin-top: 0;
            margin-bottom: 0.1rem !important;
        }
        br{
            display: none;
        }
        
        
        #page .boxnote, .qtip .boxnote, .modal .modal-body .boxnote {
            margin: 0 auto;
            width: 100%;
            text-align: left;
            border-radius: 10px;
            background: #e7e7e2;
            margin-bottom: 5px;
            position: relative;
            overflow: auto;
            font-size: 11px;
        }
        </style>";
        return true;
    }
}
