<?php

/*
 -------------------------------------------------------------------------
 DLTeams plugin for GLPI
 -------------------------------------------------------------------------
 LICENSE : This file is part of DLTeams Plugin.

 DLTeams Plugin is a GNU Free Copylefted software.
 It disallow others people than DLPlace developers to distribute, sell,
 or add additional requirements to this software.
 Though, a limited set of safe added requirements can be allowed, but
 for private or internal usage only ;  without even the implied warranty
 of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

 You should have received a copy of the GNU General Public License
 along with DLTeams Plugin. If not, see <http://www.gnu.org/licenses/>.
 --------------------------------------------------------------------------
  @package   dlteams
  @author    DLPlace developers
  @copyright Copyright (c) 2022 DLPlace
  @inspired	 DPO register plugin (Karhel Tmarr) & gdprropa (Yild)
  @license   GPLv3+ http://www.gnu.org/licenses/gpl.txt
  @link      https://github.com/dlplace/dlteams
  @since     2021
 --------------------------------------------------------------------------
 */

include_once('../../../inc/includes.php');

// Vérifier que l'utilisateur est connecté
Session::checkLoginUser();

// Valider l'ID reçu en POST
if (empty($_POST["id"]) || !is_numeric($_POST["id"])) {
    echo json_encode([
        'status' => 'error',
        'message' => 'ID invalide ou manquant',
        'data' => null
    ]);
    exit;
}

$followupId = (int) $_POST["id"];

// Charger l'objet ITILFollowup
$itilFollowup = new ITILFollowup();
if (!$itilFollowup->getFromDB($followupId)) {
    echo json_encode([
        'status' => 'error',
        'message' => 'Impossible de charger le suivi ITIL',
        'data' => null
    ]);
    exit;
}

// Traiter le contenu
$content = htmlspecialchars_decode($itilFollowup->fields["content"]);
$truncatedContent = truncateHtmlContent($content, 200);
$user = new User();
$user->getFromDB($itilFollowup->fields["users_id"]);

// Réponse JSON
$response = [
    'status' => 'success',
    'message' => 'Votre requête a été traitée avec succès',
    'data' => htmlspecialchars_decode($itilFollowup->fields["content"]),
    'data_cut' => truncateHtmlContent(htmlspecialchars_decode($itilFollowup->fields["content"]), 200),
    'author' => [
        'id' => $user->getField('id'),
        'name' => $user->getField('firstname') . ' ' . $user->getField('realname')
    ]
];

// Envoyer la réponse JSON
echo json_encode($response);


/**
 * Tronque le contenu HTML sans casser les balises.
 *
 * @param string $html   Contenu HTML à tronquer.
 * @param int    $limit  Longueur maximale du contenu tronqué.
 *
 * @return string Contenu tronqué.
 */
function truncateHtmlContent($html, $limit) {
    // Retourner une chaîne vide si le HTML est invalide
    if (empty(trim($html))) {
        return '';
    }

    $dom = new DOMDocument();

    // Charger le HTML tout en gérant les erreurs
    libxml_use_internal_errors(true);
    $dom->loadHTML('<?xml encoding="utf-8" ?>' . $html, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
    libxml_clear_errors();

    $totalLength = 0;
    $truncated = '';

    // Vérifier si le body contient des nœuds
    $body = $dom->getElementsByTagName('body')->item(0);
    if (!$body || !$body->hasChildNodes()) {
        // Si aucun nœud valide, retourner une version textuelle simplifiée
        return mb_substr(strip_tags($html), 0, $limit) . (mb_strlen(strip_tags($html)) > $limit ? '...' : '');
    }

    // Parcourir les nœuds pour tronquer le contenu
    foreach ($body->childNodes as $node) {
        if ($totalLength >= $limit) {
            break;
        }
        $truncated .= truncateHtmlNode($node, $limit - $totalLength, $totalLength);
    }

    return $truncated . ($totalLength > $limit ? '...' : '');
}

/**
 * Tronque un nœud HTML récursivement.
 *
 * @param DOMNode $node         Nœud à tronquer.
 * @param int     $limit        Limite de caractères restants.
 * @param int     &$totalLength Référence à la longueur totale comptabilisée.
 *
 * @return string Contenu tronqué du nœud.
 */
function truncateHtmlNode($node, $limit, &$totalLength) {
    if ($node->nodeType === XML_TEXT_NODE) {
        $remaining = $limit - $totalLength;
        $content = mb_substr($node->textContent, 0, $remaining);
        $totalLength += mb_strlen($content);
        return htmlspecialchars($content, ENT_QUOTES, 'UTF-8');
    } elseif ($node->nodeType === XML_ELEMENT_NODE) {
        $result = '<' . $node->nodeName;

        // Ajouter les attributs de la balise
        foreach ($node->attributes as $attribute) {
            $result .= ' ' . $attribute->nodeName . '="' . htmlspecialchars($attribute->nodeValue, ENT_QUOTES, 'UTF-8') . '"';
        }
        $result .= '>';

        // Traiter les enfants récursivement
        foreach ($node->childNodes as $child) {
            if ($totalLength >= $limit) {
                break;
            }
            $result .= truncateHtmlNode($child, $limit, $totalLength);
        }

        $result .= '</' . $node->nodeName . '>';
        return $result;
    }
    return '';
}
