<?php
/*
 -------------------------------------------------------------------------
 DLTeams plugin for GLPI
 -------------------------------------------------------------------------
 LICENSE : This file is part of DLTeams Plugin.

 DLTeams Plugin is a GNU Free Copylefted software. 
 It disallow others people than DLPlace developers to distribute, sell, 
 or add additional requirements to this software. 
 Though, a limited set of safe added requirements can be allowed, but 
 for private or internal usage only ;  without even the implied warranty 
 of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

 You should have received a copy of the GNU General Public License
 along with DLTeams Plugin. If not, see <http://www.gnu.org/licenses/>.
 --------------------------------------------------------------------------
  @package   dlteams
  @author    DLPlace developers
  @copyright Copyright (c) 2022 DLPlace
  @license   GPLv3+ http://www.gnu.org/licenses/gpl.txt
  @link      https://github.com/dlplace/dlteams
  @since     2021
 --------------------------------------------------------------------------
 */

class PluginDlteamsMultiSelector
{
    /**
     * Display a multi-selection component for any CommonDropdown itemtype
     *
     * @param string $itemtype The class name to use for data retrieval
     * @param array $options Configuration options
     * @return string|bool HTML output or false on error
     */
    public static function show($itemtype, $options = [])
    {
        global $CFG_GLPI;

        // Validate itemtype
        if (!($item = getItemForItemtype($itemtype))) {
            return false;
        }

        // Default parameters
        $params = [
            'name'           => $item->getForeignKeyField(),
            'values'         => [],
            'used'           => [],
            'return_columns' => ['id', 'name'],
            'width'          => '400px',
            'height'         => '300px',
            'entity'         => -1,
            'entity_sons'    => false,
            'condition'      => [],
            'rand'           => mt_rand(),
            'display'        => true,
            'placeholder'    => __('Rechercher...', 'dlteams'),
            'emptylabel'     => __('Aucun élément sélectionné', 'dlteams'),
            'addicon'        => false,
            'url'            => $CFG_GLPI['root_doc'] . "/marketplace/dlteams/ajax/multiselector_data.php"
        ];

        // Merge with provided options
        if (is_array($options) && count($options)) {
            foreach ($options as $key => $val) {
                $params[$key] = $val;
            }
        }


        // Sanitize name
        $params['name'] = Html::sanitizeInputName($params['name']);

        // Generate component HTML
        $output = self::renderComponent($itemtype, $params);

        if ($params['display']) {
            echo $output;
            return true;
        }

        return $output;
    }

    /**
     * Get items from database for the specified itemtype
     *
     * @param string $itemtype The class name
     * @param array $params Parameters for filtering
     * @return array Array of items
     */
    public static function getItems($itemtype, $params = [])
    {
        global $DB;

        if (!($item = getItemForItemtype($itemtype))) {
            return [];
        }

        $table = $item->getTable();
        $where = [];

        // Basic filters
        if ($item->maybeDeleted()) {
            $where["$table.is_deleted"] = 0;
        }
        if ($item->maybeTemplate()) {
            $where["$table.is_template"] = 0;
        }

        // Entity filter
        if (isset($params['entity']) && $params['entity'] >= 0) {
            if ($item->isEntityAssign()) {
                $where["$table.entities_id"] = $params['entity'];
                if (isset($params['entity_sons']) && $params['entity_sons']) {
                    $entities = getSonsOf('glpi_entities', $params['entity']);
                    $where["$table.entities_id"] = $entities;
                }
            }
        }

        // Additional conditions
        if (isset($params['condition']) && is_array($params['condition'])) {
            $where = array_merge($where, $params['condition']);
        }

        // Search filter
        if (isset($params['searchText']) && strlen($params['searchText']) > 0) {
            $search = Search::makeTextSearchValue($params['searchText']);
            $where['OR'] = [
                ["$table.name" => ['LIKE', $search]]
            ];
            
            // Add additional searchable fields if they exist
            if ($item->isField('comment')) {
                $where['OR'][] = ["$table.comment" => ['LIKE', $search]];
            }
        }

        // Exclude used items
        if (isset($params['used']) && is_array($params['used']) && count($params['used']) > 0) {
            $where["$table.id"] = ['NOT IN', $params['used']];
        }

        // Build select fields based on return_columns
        $select_fields = [];
        foreach ($params['return_columns'] as $column) {
            if ($item->isField($column)) {
                $select_fields[] = "$table.$column";
            }
        }

        if (empty($select_fields)) {
            $select_fields = ["$table.id", "$table.name"];
        }

        // Execute query
        $iterator = $DB->request([
            'SELECT' => $select_fields,
            'FROM'   => $table,
            'WHERE'  => $where,
            'ORDER'  => "$table.name ASC",
            'LIMIT'  => 1000 // Prevent excessive results
        ]);

        $items = [];
        foreach ($iterator as $data) {
            $items[] = $data;
        }

        return $items;
    }

    /**
     * Render the complete multi-selector component
     *
     * @param string $itemtype The class name
     * @param array $params Configuration parameters
     * @return string HTML output
     */
    private static function renderComponent($itemtype, $params)
    {
        $rand = $params['rand'];
        $name = $params['name'];
        $width = $params['width'];
        $height = $params['height'];

        // Include CSS and JS
        $output = self::includeAssets();

        // Container
        $output .= "<div class='multiselector-container' id='multiselector_$rand' style='width: $width;'>";
        
        // Search box
        $output .= self::renderSearchBox($rand, $params);
        
        // Items list container
        $output .= "<div class='multiselector-list' style='height: $height;' id='multiselector_list_$rand'>";
        $output .= "<div class='multiselector-loading'>" . __('Chargement...', 'dlteams') . "</div>";
        $output .= "</div>";
        
        // Action buttons
        $output .= self::renderActionButtons($rand);
        
        // Hidden input for selected values
        $selected_values = is_array($params['values']) ? json_encode($params['values']) : '[]';
        $output .= "<input type='hidden' name='$name' id='multiselector_values_$rand' value='" . Html::entities_deep($selected_values) . "' />";
        
        $output .= "</div>";

        // JavaScript initialization
        $output .= self::renderJavaScript($itemtype, $params, $rand);

        return $output;
    }

    /**
     * Render the search box
     *
     * @param int $rand Random ID
     * @param array $params Parameters
     * @return string HTML output
     */
    private static function renderSearchBox($rand, $params)
    {
        $placeholder = $params['placeholder'];
        
        $output = "<div class='multiselector-search'>";
        $output .= "<input type='text' class='form-control' id='multiselector_search_$rand' placeholder='$placeholder' />";
        $output .= "</div>";
        
        return $output;
    }

    /**
     * Render action buttons
     *
     * @param int $rand Random ID
     * @return string HTML output
     */
    private static function renderActionButtons($rand)
    {
        $output = "<div class='multiselector-actions'>";
        $output .= "<button type='button' class='btn btn-sm btn-outline-secondary' id='multiselector_select_all_$rand'>";
        $output .= "<i class='fas fa-check-square'></i> " . __('Tout sélectionner', 'dlteams');
        $output .= "</button>";
        $output .= "<button type='button' class='btn btn-sm btn-outline-secondary' id='multiselector_clear_$rand'>";
        $output .= "<i class='fas fa-times'></i> " . __('Effacer', 'dlteams');
        $output .= "</button>";
        $output .= "</div>";
        
        return $output;
    }

    /**
     * Include CSS and JS assets
     *
     * @return string HTML output
     */
    private static function includeAssets()
    {
        global $CFG_GLPI;
        
        $output = "<link rel='stylesheet' type='text/css' href='" . $CFG_GLPI['root_doc'] . "/marketplace/dlteams/css/multiselector.css' />";
        $output .= "<script type='text/javascript' src='" . $CFG_GLPI['root_doc'] . "/marketplace/dlteams/js/multiselector.js'></script>";
        
        return $output;
    }

    /**
     * Render JavaScript initialization
     *
     * @param string $itemtype The class name
     * @param array $params Parameters
     * @param int $rand Random ID
     * @return string JavaScript code
     */
    private static function renderJavaScript($itemtype, $params, $rand)
    {
        $url = $params['url'];
        $return_columns = json_encode($params['return_columns']);
        $entity = $params['entity'];
        $condition = json_encode($params['condition']);
        $selected_values = json_encode($params['values']);

        $output = "<script type='text/javascript'>";
        $output .= "$(document).ready(function() {";
        $output .= "  if (typeof PluginDlteamsMultiSelector !== 'undefined') {";
        $output .= "    PluginDlteamsMultiSelector.init({";
        $output .= "      rand: '$rand',";
        $output .= "      itemtype: '$itemtype',";
        $output .= "      url: '$url',";
        $output .= "      return_columns: $return_columns,";
        $output .= "      entity: $entity,";
        $output .= "      condition: $condition,";
        $output .= "      selected_values: $selected_values";
        $output .= "    });";
        $output .= "  }";
        $output .= "});";
        $output .= "</script>";

        return $output;
    }
}

