<?php

class PluginDlteamsRelationManager {
    private $db;
    private array $post;

    public function __construct(array $postData) {
        global $DB;
        $this->db   = $DB;
        $this->post = $postData;
    }

    public function handleLinkElement(): void {
        // 1) Vérifications préalables
        if (!isset($this->post['link_element']) || empty($this->post['items_id'])) {
            Session::addMessageAfterRedirect(__('Veuillez choisir un élément à relier'), false, ERROR);
            Html::back();
            return;
        }

        $this->db->beginTransaction();

        try {
            // 2) Préparer et exécuter les deux inserts
            $i2 = $this->buildSecondRelation();
            $this->db->insert($this->getItemTable($this->post['itemtype1']), $i2);

            $i1 = $this->buildFirstRelation();
            $this->db->insert($this->getItemTable($this->post['itemtype']), $i1);

            // 3) Appliquer sur enfants éventuels
            if (!empty($this->post['apply_on_childs'])) {
                $this->applyOnChilds();
            }

            $this->db->commit();
            Session::addMessageAfterRedirect(__('Ajouté avec succès'));
        } catch (\Throwable $e) {
            $this->db->rollBack();
            if (Session::DEBUG_MODE) {
                Session::addMessageAfterRedirect($e->getMessage(), false, ERROR);
            }
            Session::addMessageAfterRedirect(__('Une erreur est survenue lors de la création de la relation'), false, ERROR);
            Html::back();
        }
    }

    private function buildSecondRelation(): array {
        // Déterminer le nom de colonne cible (items_id, entities_id, auditcategories_id…)
        $col       = $this->determineColumn($this->post['itemtype1']);
        $relation  = [
            $col             => $this->post['items_id1'],
            'items_id'       => $this->post['items_id'],
            'itemtype'       => $this->post['itemtype'],
        ];

        // Ajouter facultativement ‘comment’, ‘mandatory’, ‘is_directory’, etc.
        if ($this->fieldExists($this->post['itemtype1'], 'comment')) {
            $relation['comment'] = $this->post['comment'] ?? '';
        }
        if (!empty($this->post['mandatory']) && $this->post['transformdocument'] ?? false) {
            $relation['document_mandatory'] = $this->post['mandatory'];
        }
        if (!empty($this->post['is_directory']) && $this->post['itemtype1'] === DataCatalog::class) {
            $relation['is_directory'] = $this->post['is_directory'];
        }

        // Cas spéciaux AccountKey↔User/Group
        if ($this->isAccountKeyWithUserOrGroup()) {
            $relation = array_merge($relation, $this->buildAccountKeyRelationData());
        }

        return $relation;
    }

    private function buildFirstRelation(): array {
        $col      = $this->determineColumn($this->post['itemtype'], true);
        $keyField = $this->post['processitemtype1'] ? 'itemtype1' : 'itemtype';
        $idField  = $this->post['processitemtype1'] ? 'items_id1' : 'items_id';

        $relation = [
            $col          => $this->post['items_id'],
            $idField      => $this->post['items_id1'],
            $keyField     => $this->post['itemtype1'],
            'comment'     => $this->post['comment'] ?? '',
        ];

        if (!empty($this->post['is_directory']) && $this->post['itemtype'] === \PluginDlteamsAccountKey::class) {
            $relation['is_directory'] = $this->post['is_directory'];
        }

        return $relation;
    }

    private function applyOnChilds(): void {
        $children = $this->db->request(PluginDlteamsDataCatalog::getTable(), [
            'plugin_dlteams_datacatalogs_id' => $this->post['items_id1']
        ]);
        foreach ($children as $child) {
            // Construire array1 et array2
            $array1 = [
                // identique à buildFirstRelation(), mais 'items_id' => $child['id']
            ];
            $array2 = [
                // identique à buildSecondRelation(), mais 'items_id1' => $child['id']
            ];
            // Insertion conditionnelle
            $this->insertIfNotExists($this->getItemTable($this->post['itemtype1']), $array1);
            $this->insertIfNotExists($this->getItemTable($this->post['itemtype']),  $array2);
        }
    }

    /**
     * Détermine le nom de la colonne à utiliser pour la relation.
     *
     * @param string $class Nom complet de la classe (itemtype ou itemtype1).
     * @param bool   $first Si true, on calcule la colonne pour la première relation (i1), sinon pour la seconde (i2).
     * @return string Nom de la colonne (ex. "items_id", "entities_id", "auditcategories_id", etc.).
     */
    private function determineColumn(string $class, bool $first = false): string {
        if (! $first) {
            // Seconde relation (items_id1 ↔ items_id)
            $col = strtolower(str_replace("PluginDlteams", "", $class)) . 's_id';

            if ($class === PluginDlteamsAuditCategory::class) {
                $col = 'auditcategories_id';
            } elseif ($class === Entity::class) {
                $col = 'entities_id';
            }

            return $col;
        }

        // Première relation (items_id ↔ items_id1)
        return match ($class) {
            PluginDlteamsThirdPartyCategory::class   => 'thirdpartycategories_id',
            PluginDlteamsActivitycategory::class     => 'activitycategories_id',
            PluginDlteamsRightMeasureCategory::class => 'rightmeasurecategories_id',
            PluginDlteamsProcessedDataCategory::class=> 'processeddatacategories_id',
            PluginDlteamsPolicieFormCategory::class  => 'policieformcategories_id',
            default => strtolower(str_replace("PluginDlteams", "", $class)) . 's_id',
        };
    }


    /**
     * Retourne le nom de la table associée à la classe Item correspondant à $itemtype.
     *
     * @param string $itemtype Nom de la classe de base (ex. PluginDlteamsRecord, Appliance, etc.)
     * @return string Nom de la table en base de données
     */
    private function getItemTable(string $itemtype): string {
        // Cas spécial : Appliance → PluginDlteamsAppliance
        if ($itemtype === Appliance::class) {
            $itemtype = PluginDlteamsAppliance::class;
        }

        // Récupère la liste des exceptions (où on préfixe par "Item_")
        $exceptions = PluginDlteamsUtils::itemtypeExceptionList();

        if (in_array($itemtype, $exceptions, true)) {
            $itemClass = 'Item_' . $itemtype;
        }
        // Si une classe "<itemtype>_Item" existe, on l'utilise
        elseif (class_exists($itemtype . '_Item')) {
            $itemClass = $itemtype . '_Item';
        }
        // Sinon, on tombe sur le fallback "PluginDlteams<itemtype>_Item"
        else {
            $itemClass = 'PluginDlteams' . $itemtype . '_Item';
        }

        // Appelle la méthode statique pour récupérer le nom de table
        return $itemClass::getTable();
    }


    private function fieldExists(string $class, string $field): bool {
        return $this->db->fieldExists($class::getTable(), $field);
    }

    private function isAccountKeyWithUserOrGroup(): bool {
        return ($this->post['itemtype1'] === PluginDlteamsAccountKey::class)
            && in_array($this->post['itemtype'], [User::class, Group::class], true);
    }

    private function buildAccountKeyRelationData(): array {
        $account = new PluginDlteamsAccountKey();
        $account->getFromDB($this->post['items_id1']);
        return [
            ($this->post['itemtype'] === User::class)  ? 'users_id' : 'groups_id'
            => $this->post['items_id'],
            'name' => $account->fields['name']
        ];
    }

    /**
     * Retourne le nom de la classe “Item” correspondant à $itemtype.
     *
     * @param string $itemtype Nom de la classe de base (ex. PluginDlteamsRecord, Appliance, etc.)
     * @return string Nom de la classe à instancier (ex. PluginDlteamsRecord_Item, Item_PluginDlteamsXxx, etc.)
     */
    private function getItemClass(string $itemtype): string {
        // Cas spécial : Appliance → PluginDlteamsAppliance
        if ($itemtype === Appliance::class) {
            $itemtype = PluginDlteamsAppliance::class;
        }

        // Récupère la liste des exceptions (où on préfixe par "Item_")
        $exceptions = PluginDlteamsUtils::itemtypeExceptionList();

        if (in_array($itemtype, $exceptions, true)) {
            // ex. Item_PluginDlteamsXxx
            return 'Item_' . $itemtype;
        }

        // Si une classe "<itemtype>_Item" existe, on l'utilise
        if (class_exists($itemtype . '_Item')) {
            return $itemtype . '_Item';
        }

        // Sinon, fallback sur "PluginDlteams<itemtype>_Item"
        return 'PluginDlteams' . $itemtype . '_Item';
    }


    private function insertIfNotExists(string $table, array $data): void {
//        $item = new $this->getItemClassByTable($table);
        $crit = $data;
        $iterator = $this->db->request([
            "FROM" => $table,
            "WHERE" => $crit
        ]);

        if (! count($iterator)) {
            $this->db->insert($table, $data);
        }
    }
}
