/**
 * -------------------------------------------------------------------------
 * DLTeams plugin for GLPI
 * -------------------------------------------------------------------------
 * LICENSE : This file is part of DLTeams Plugin.
 *
 * DLTeams Plugin is a GNU Free Copylefted software. 
 * It disallow others people than DLPlace developers to distribute, sell, 
 * or add additional requirements to this software. 
 * Though, a limited set of safe added requirements can be allowed, but 
 * for private or internal usage only ;  without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * You should have received a copy of the GNU General Public License
 * along with DLTeams Plugin. If not, see <http://www.gnu.org/licenses/>.
 * --------------------------------------------------------------------------
 *  @package   dlteams
 *  @author    DLPlace developers
 *  @copyright Copyright (c) 2022 DLPlace
 *  @license   GPLv3+ http://www.gnu.org/licenses/gpl.txt
 *  @link      https://github.com/dlplace/dlteams
 *  @since     2021
 * --------------------------------------------------------------------------
 */


var PluginDlteamsMultiSelector = {
    instances: {},
    
    /**
     * Initialize a multi-selector instance
     * @param {Object} config Configuration object
     */
    init: function(config) {
        var self = this;
        var rand = config.rand;
        
        // Store instance configuration
        this.instances[rand] = {
            config: config,
            selectedValues: config.selected_values || [],
            allItems: [],
            filteredItems: []
        };
        
        // Load initial data
        this.loadData(rand);
        
        // Bind events
        this.bindEvents(rand);
    },
    
    /**
     * Load data from server
     * @param {string} rand Instance ID
     * @param {string} searchText Optional search text
     */
    loadData: function(rand, searchText) {
        var self = this;
        var instance = this.instances[rand];
        var config = instance.config;
        
        // Show loading
        $('#multiselector_list_' + rand).html('<div class="multiselector-loading"><i class="fas fa-spinner fa-spin"></i> Chargement...</div>');
        
        // Prepare request data
        var requestData = {
            itemtype: config.itemtype,
            return_columns: JSON.stringify(config.return_columns),
            entity: config.entity,
            condition: JSON.stringify(config.condition)
        };
        
        if (searchText) {
            requestData.searchText = searchText;
        }
        
        // AJAX request
        $.ajax({
            url: config.url,
            type: 'POST',
            data: requestData,
            dataType: 'json',
            success: function(response) {
                if (response.success) {
                    instance.allItems = response.items;
                    instance.filteredItems = response.items;
                    self.renderItems(rand);
                } else {
                    self.showError(rand, response.error || 'Erreur lors du chargement des données');
                }
            },
            error: function(xhr, status, error) {
                self.showError(rand, 'Erreur de communication avec le serveur');
            }
        });
    },
    
    /**
     * Render items list
     * @param {string} rand Instance ID
     */
    renderItems: function(rand) {
        var instance = this.instances[rand];
        var items = instance.filteredItems;
        var selectedValues = instance.selectedValues;
        
        var html = '';
        
        if (items.length === 0) {
            html = '<div class="multiselector-empty">Aucun élément trouvé</div>';
        } else {
            items.forEach(function(item) {
                var isSelected = selectedValues.indexOf(item.id.toString()) !== -1;
                var checkedAttr = isSelected ? 'checked' : '';
                
                html += '<div class="multiselector-item" data-id="' + item.id + '">';
                html += '<label class="multiselector-label">';
                html += '<input type="checkbox" class="multiselector-checkbox" value="' + item.id + '" ' + checkedAttr + ' />';
                html += '<span class="multiselector-text">' + this.escapeHtml(item.name || item.id) + '</span>';
                html += '</label>';
                html += '</div>';
            }.bind(this));
        }
        
        $('#multiselector_list_' + rand).html(html);
        
        // Update counter
        this.updateCounter(rand);
    },
    
    /**
     * Bind events for an instance
     * @param {string} rand Instance ID
     */
    bindEvents: function(rand) {
        var self = this;
        
        // Search input
        $('#multiselector_search_' + rand).on('input', function() {
            var searchText = $(this).val();
            self.filterItems(rand, searchText);
        });
        
        // Checkbox changes
        $(document).on('change', '#multiselector_list_' + rand + ' .multiselector-checkbox', function() {
            var value = $(this).val();
            var isChecked = $(this).is(':checked');
            
            self.toggleSelection(rand, value, isChecked);
        });
        
        // Select all button
        $('#multiselector_select_all_' + rand).on('click', function() {
            self.selectAll(rand);
        });
        
        // Clear button
        $('#multiselector_clear_' + rand).on('click', function() {
            self.clearSelection(rand);
        });
    },
    
    /**
     * Filter items based on search text
     * @param {string} rand Instance ID
     * @param {string} searchText Search text
     */
    filterItems: function(rand, searchText) {
        var instance = this.instances[rand];
        
        if (!searchText || searchText.length < 2) {
            // If search is too short, reload all data
            this.loadData(rand);
            return;
        }
        
        // Load filtered data from server
        this.loadData(rand, searchText);
    },
    
    /**
     * Toggle selection of an item
     * @param {string} rand Instance ID
     * @param {string} value Item ID
     * @param {boolean} isSelected Whether item is selected
     */
    toggleSelection: function(rand, value, isSelected) {
        var instance = this.instances[rand];
        var selectedValues = instance.selectedValues;
        
        if (isSelected) {
            // Add to selection if not already present
            if (selectedValues.indexOf(value) === -1) {
                selectedValues.push(value);
            }
        } else {
            // Remove from selection
            var index = selectedValues.indexOf(value);
            if (index !== -1) {
                selectedValues.splice(index, 1);
            }
        }
        
        // Update hidden input
        this.updateHiddenInput(rand);
        
        // Update counter
        this.updateCounter(rand);
    },
    
    /**
     * Select all visible items
     * @param {string} rand Instance ID
     */
    selectAll: function(rand) {
        var instance = this.instances[rand];
        
        // Get all visible checkboxes and check them
        $('#multiselector_list_' + rand + ' .multiselector-checkbox').each(function() {
            var value = $(this).val();
            if (instance.selectedValues.indexOf(value) === -1) {
                instance.selectedValues.push(value);
            }
            $(this).prop('checked', true);
        });
        
        // Update hidden input
        this.updateHiddenInput(rand);
        
        // Update counter
        this.updateCounter(rand);
    },
    
    /**
     * Clear all selections
     * @param {string} rand Instance ID
     */
    clearSelection: function(rand) {
        var instance = this.instances[rand];
        
        // Clear selected values
        instance.selectedValues = [];
        
        // Uncheck all checkboxes
        $('#multiselector_list_' + rand + ' .multiselector-checkbox').prop('checked', false);
        
        // Update hidden input
        this.updateHiddenInput(rand);
        
        // Update counter
        this.updateCounter(rand);
    },
    
    /**
     * Update the hidden input with selected values
     * @param {string} rand Instance ID
     */
    updateHiddenInput: function(rand) {
        var instance = this.instances[rand];
        var selectedValues = instance.selectedValues;
        
        $('#multiselector_values_' + rand).val(JSON.stringify(selectedValues));
    },
    
    /**
     * Update the selection counter
     * @param {string} rand Instance ID
     */
    updateCounter: function(rand) {
        var instance = this.instances[rand];
        var count = instance.selectedValues.length;
        
        // Update button text
        var selectAllBtn = $('#multiselector_select_all_' + rand);
        var clearBtn = $('#multiselector_clear_' + rand);
        
        if (count > 0) {
            clearBtn.html('<i class="fas fa-times"></i> Effacer (' + count + ')');
            clearBtn.removeClass('btn-outline-secondary').addClass('btn-outline-danger');
        } else {
            clearBtn.html('<i class="fas fa-times"></i> Effacer');
            clearBtn.removeClass('btn-outline-danger').addClass('btn-outline-secondary');
        }
    },
    
    /**
     * Show error message
     * @param {string} rand Instance ID
     * @param {string} message Error message
     */
    showError: function(rand, message) {
        var html = '<div class="multiselector-error"><i class="fas fa-exclamation-triangle"></i> ' + message + '</div>';
        $('#multiselector_list_' + rand).html(html);
    },
    
    /**
     * Escape HTML characters
     * @param {string} text Text to escape
     * @return {string} Escaped text
     */
    escapeHtml: function(text) {
        var div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
};

