<?php
/*
 -------------------------------------------------------------------------
 DLTeams plugin for GLPI
 -------------------------------------------------------------------------
*/

class PluginDlteamsITILFollowup_Reaction extends CommonDBTM {

    public static $rightname = 'plugin_dlteams:itilfollowup_reaction';

    public static function getTable($classname = null) {
        return 'glpi_plugin_dlteams_itilfollowups_reactions';
    }

    /**
     * Vérifie l’accès du courant au message visé (existence + entité).
     */
    public static function checkMessageAccessOrDie(int $itilfollowups_id): void {
        $f = new ITILFollowup();
        if (!$f->getFromDB($itilfollowups_id)) {
            Html::displayErrorAndDie(__('Message introuvable'));
        }
        if (Session::isMultiEntitiesMode()) {
            $ent = (int)($f->fields['entities_id'] ?? 0);
            if (!Session::haveAccessToEntity($ent)) {
                Html::displayErrorAndDie(__('Accès refusé'));
            }
        }
    }

    /**
     * Toggle d’une réaction (ajout si absente, suppression si présente).
     * @return array{hasReacted:bool,newCount:int}
     */
    public static function toggle(
        int $itilfollowups_id,
        string $emoji_key,
        ?int $users_id = null,
        string $reactor_itemtype = User::class
    ): array {
        global $DB;

        self::checkMessageAccessOrDie($itilfollowups_id);

        $users_id  = $users_id ?? (int)Session::getLoginUserID();
        $emoji_key = trim((string)$emoji_key);

        if ($users_id <= 0) {
            Html::displayErrorAndDie(__('Utilisateur non connecté'));
        }
        if ($emoji_key === '') {
            Html::displayErrorAndDie(__('Emoji invalide'));
        }

        // Sélecteur unique conforme à l’index uniq_message_user_emoji
        $selector = [
            'itilfollowups_id' => $itilfollowups_id,
            'itemtype'         => $reactor_itemtype,
            'items_id'         => $users_id,
            'emoji_key'        => $emoji_key,
        ];

        // Existence ?
        $exists = false;
        $iter = $DB->request([
            'FROM'  => self::getTable(),
            'WHERE' => $selector,
            'LIMIT' => 1
        ]);
        if ($iter && count($iter) > 0) {
            $exists = true;
        }

        if ($exists) {
            // Supprimer
            $DB->delete(self::getTable(), $selector);
            $hasReacted = false;
        } else {
            // Insérer
            $DB->insert(self::getTable(), $selector + [
                    'date_creation' => new \QueryExpression('NOW()'),
                    'date_mod'      => new \QueryExpression('NOW()'),
                ]);
            $hasReacted = true;
        }

        // Recompter pour cet emoji sur ce message
        $cnt = 0;
        $sql = "SELECT COUNT(*) AS c
                FROM " . self::getTable() . "
               WHERE itilfollowups_id = " . (int)$itilfollowups_id . "
                 AND emoji_key = '" . $DB->escape($emoji_key)."'";
        foreach ($DB->query($sql) as $row) {
            $cnt = (int)$row['c'];
        }

        // Marquer le message "vu"
        self::ensureView($itilfollowups_id, $users_id);

        return ['hasReacted' => $hasReacted, 'newCount' => $cnt];
    }

    /**
     * Marque un message comme "vu" par l’utilisateur (si pas déjà fait).
     */
    public static function ensureView(int $itilfollowups_id, int $users_id): void {
        global $DB;
        $t = PluginDlteamsITILFollowup_View::getTable();

        $crit = [
            'FROM'  => $t,
            'WHERE' => [
                'itilfollowups_id' => $itilfollowups_id,
                'itemtype'         => User::class,
                'items_id'         => $users_id
            ],
            'LIMIT' => 1
        ];
        $it = $DB->request($crit);
        if (!$it || count($it) === 0) {
            $DB->insert($t, [
                'itilfollowups_id' => $itilfollowups_id,
                'itemtype'         => User::class,
                'items_id'         => $users_id,
                'date_creation'    => new \QueryExpression('NOW()'),
                'date_mod'         => new \QueryExpression('NOW()')
            ]);
        }
    }

    /**
     * Agrégats par message :
     *  [
     *    <msg_id> => [
     *      ['emoji_key'=>'👍','count'=>N,'reacted_by_me'=>bool], ...
     *    ],
     *    ...
     *  ]
     */
    public static function getAggregatesForMessages(
        array $message_ids,
        ?int $users_id = null,
        string $reactor_itemtype = User::class
    ): array {
        global $DB;

        $users_id = $users_id ?? (int)Session::getLoginUserID();

        // Normaliser la liste d’IDs
        $message_ids = array_values(array_unique(array_map('intval', $message_ids)));
        if (empty($message_ids)) {
            return [];
        }
        $in = '(' . implode(',', $message_ids) . ')';

        // Comptages par emoji
        $sqlCounts = "
         SELECT itilfollowups_id, emoji_key, COUNT(*) AS cnt
           FROM " . self::getTable() . "
          WHERE itilfollowups_id IN $in
          GROUP BY itilfollowups_id, emoji_key
      ";

        $byMsg = [];
        foreach ($DB->query($sqlCounts) as $row) {
            $mid   = (int)$row['itilfollowups_id'];
            $emoji = (string)$row['emoji_key'];
            $cnt   = (int)$row['cnt'];
            if (!isset($byMsg[$mid])) $byMsg[$mid] = [];
            $byMsg[$mid][$emoji] = ['count' => $cnt, 'reacted_by_me' => false];
        }

        // Mes réactions (pour le flag reacted_by_me)
        if ($users_id > 0) {
            $sqlMine = "
            SELECT itilfollowups_id, emoji_key
              FROM " . self::getTable() . "
             WHERE itilfollowups_id IN $in
               AND itemtype = '" . $DB->escape($reactor_itemtype) . "'
               AND items_id = " . (int)$users_id;
            foreach ($DB->query($sqlMine) as $row) {
                $mid   = (int)$row['itilfollowups_id'];
                $emoji = (string)$row['emoji_key'];
                if (!isset($byMsg[$mid])) {
                    $byMsg[$mid] = [];
                }
                if (!isset($byMsg[$mid][$emoji])) {
                    $byMsg[$mid][$emoji] = ['count' => 0, 'reacted_by_me' => true];
                } else {
                    $byMsg[$mid][$emoji]['reacted_by_me'] = true;
                }
            }
        }

        // Mise en forme finale
        $out = [];
        foreach ($message_ids as $mid) {
            $out[$mid] = [];
            if (!empty($byMsg[$mid])) {
                foreach ($byMsg[$mid] as $emoji => $info) {
                    $out[$mid][] = [
                        'emoji_key'     => $emoji,
                        'count'         => (int)$info['count'],
                        'reacted_by_me' => (bool)$info['reacted_by_me'],
                    ];
                }
            }
        }
        return $out;
    }

    /**
     * Liste paginée des utilisateurs ayant réagi avec un emoji donné.
     * Retourne ['total'=>int,'rows'=>[{'users_id','name','initials','bg','date_creation'},...]]
     */
    public static function whoReacted(
        int $itilfollowups_id,
        string $emoji_key,
        int $limit = 20,
        int $offset = 0,
        string $reactor_itemtype = User::class
    ): array {
        global $DB;

        self::checkMessageAccessOrDie($itilfollowups_id);

        $emoji_key = trim((string)$emoji_key);
        if ($emoji_key === '') {
            return ['total' => 0, 'rows' => []];
        }

        $table = self::getTable();

        // Total
        $sqlCount = "
         SELECT COUNT(*) AS c
           FROM $table
          WHERE itilfollowups_id = " . (int)$itilfollowups_id . "
            AND emoji_key        = '" . $DB->escape($emoji_key) . "'
            AND itemtype         = '" . $DB->escape($reactor_itemtype)."'";
        $total = 0;
        foreach ($DB->query($sqlCount) as $r) {
            $total = (int)$r['c'];
        }

        // Lignes
        $sql = "
         SELECT items_id, date_creation
           FROM $table
          WHERE itilfollowups_id = " . (int)$itilfollowups_id . "
            AND emoji_key        = '" . $DB->escape($emoji_key) . "'
            AND itemtype         = '" . $DB->escape($reactor_itemtype) . "'
          ORDER BY date_creation DESC
          LIMIT " . (int)$limit . " OFFSET " . (int)$offset;

        $rows = [];
        foreach ($DB->query($sql) as $r) {
            $u = new User();
            $u->getFromDB((int)$r['items_id']);
            $rows[] = [
                'users_id'      => (int)$r['items_id'],
                'name'          => trim(($u->fields['firstname'] ?? '') . ' ' . ($u->fields['realname'] ?? '')),
                'initials'      => $u->getUserInitials(),
                'bg'            => $u->getUserInitialsBgColor(),
                'date_creation' => $r['date_creation']
            ];
        }

        return ['total' => $total, 'rows' => $rows];
    }
}
